﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using dataladder.Matching;

namespace UsingTheExistingProjectWithGUI
{
  public partial class MainForm : Form
  {
    #region Fields

    EngineWrapper engineWrapper = null;
    string engineConfigurationIniFileName = "configuration.ini";
    string projectsIniFileName = "projects.ini";
    string projectName = "";
    string appPath;
    IniParser projectsParser;

    string[] previousSearchValues = new string[0];
    bool doTimerSearch = false;

    #endregion

    #region Constructors

    public MainForm()
    {
      InitializeComponent();
      appPath = System.IO.Path.GetDirectoryName(Application.ExecutablePath);
      if (!System.IO.File.Exists(projectsIniFileName))
      {
        using (System.IO.File.Create(projectsIniFileName)) { }
      }
      string fullIniName = System.IO.Path.Combine(appPath, projectsIniFileName);
      projectsParser = new IniParser(fullIniName);
    }

    #endregion

    #region Properties and Fields

    bool autoSearchAndLoad { get { return autoSaveCheckBox.Checked; } }

    EngineWrapper.AddWeightToFirstLetterActions addWeightToFirstLetterAction
    {
      get
      {
        EngineWrapper.AddWeightToFirstLetterActions result;
        if (projectRadioButton.Checked)
        {
          result = EngineWrapper.AddWeightToFirstLetterActions.Project;
        }
        else if (onRadioButton.Checked)
        {
          result = EngineWrapper.AddWeightToFirstLetterActions.On;
        }
        else
        {
          result = EngineWrapper.AddWeightToFirstLetterActions.Off;
        }
        return result;
      }
    }

    #endregion

    #region Methods

    private void openProject()
    {
      string fullIniName = System.IO.Path.Combine(appPath, engineConfigurationIniFileName);
      IniParser parser = new IniParser(fullIniName);
      string projectsPath = parser.GetSetting("AppSettings", "projectsPath");
      projectOpenFileDialog.InitialDirectory = projectsPath;
      if (projectOpenFileDialog.ShowDialog() == System.Windows.Forms.DialogResult.OK)
      {
        string projectFileName = projectOpenFileDialog.FileName;
        projectName = System.IO.Path.GetFileNameWithoutExtension(projectFileName);
        projectNameTextBox.Text = projectFileName;
        if (engineWrapper != null)
        {
          engineWrapper.Dispose();
          engineWrapper = null;
        }
        engineWrapper = new EngineWrapper(projectFileName);
        if (engineWrapper.DataSourceCount != 1)
        {
          engineWrapper.Dispose();
          MessageBox.Show("Only projects with one data source are supported");
          searchButton.Enabled = false;
          return;
        }
        searchPanel.Controls.Clear();
        const int verticalDist = 23;
        const int verticalMargin = 3;
        const int horizontalMargin = 3;
        int maxLeft = 0;
        for (int i = 0; i < engineWrapper.MatchingFieldNames.Length; i++)
        {
          string fieldName = engineWrapper.MatchingFieldNames[i];
          Label label = new Label();
          label.Width = 20;
          label.AutoSize = true;
          label.Name = fieldName;
          label.Text = fieldName + ":";
          label.Top = i * verticalDist + verticalMargin;
          label.Left = horizontalMargin;
          searchPanel.Controls.Add(label);
          maxLeft = Math.Max(label.Left + label.Width, maxLeft);
        }
        maxLeft += horizontalMargin;
        for (int i = 0; i < engineWrapper.MatchingFieldNames.Length; i++)
        {
          string fieldName = engineWrapper.MatchingFieldNames[i];
          TextBox textBox = new TextBox();
          textBox.Name = fieldName;
          textBox.Top = i * verticalDist + verticalMargin;
          textBox.Left = maxLeft;
          textBox.Width = searchPanel.Width - maxLeft - horizontalMargin;
          if (autoSearchAndLoad)
          {
            string value = projectsParser.GetSetting(projectName, fieldName);
            textBox.Text = value;
          }
          searchPanel.Controls.Add(textBox);
        }
        searchButton.Enabled = true;
      }
    }

    string[] getSearchValues()
    {
      List<string> result = new List<string>();
      for (int i = 0; i < searchPanel.Controls.Count; i++)
      {
        Control control = searchPanel.Controls[i];
        if (control is TextBox)
        {
          TextBox textBox = control as TextBox;
          result.Add(textBox.Text.Trim());
        }
      }
      return result.ToArray();
    }

    private void saveSearchValues(string[] values)
    {
      if (autoSearchAndLoad)
      {
        for (int i = 0; i < values.Length; i++)
        {
          string value = values[i];
          projectsParser.AddSetting(projectName, engineWrapper.MatchingFieldNames[i], value);
        }
        projectsParser.SaveSettings();
      }
    }

    #endregion

    #region Events and Handlers

    private void openProjectButton_Click(object sender, EventArgs e)
    {
      openProject();
    }

    private void searchButton_Click(object sender, EventArgs e)
    {
      search();
    }

    private void search()
    {
      if (engineWrapper != null)
      {
        timeLabel.Text = "...";
        Application.DoEvents();
        System.Diagnostics.Stopwatch stopwatch = new System.Diagnostics.Stopwatch();
        stopwatch.Start();
        string[] values = getSearchValues();
        int bestMatchesCapacity;
        if (!int.TryParse(maxCapacityTextBox.Text, out bestMatchesCapacity))
        {
          MessageBox.Show("invalid max capacity!");
          return;
        }
        dataladder.Data.OnDriveTable searchResultsTable = engineWrapper.FindMatches(values, bestMatchesCapacity, addWeightToFirstLetterAction);
        stopwatch.Stop();
        timeLabel.Text = "response time: " + stopwatch.Elapsed.ToString();
        searchResultsDataGridView.DataSource = null;
        searchResultsDataGridView.DataSource = new dataladder.XtraGridHelper.VirtualListDynamic(searchResultsTable);
        saveSearchValues(values);
      }
    }

    #endregion

    private void searchTimer_Tick(object sender, EventArgs e)
    {
      if (engineWrapper != null)
      {
        string[] searchValues = getSearchValues();
        if (searchValues.Length == previousSearchValues.Length)
        {
          for (int i = 0; i < searchValues.Length; i++)
          {
            if (searchValues[i] != previousSearchValues[i])
            {
              doTimerSearch = true;
            }
            previousSearchValues[i] = searchValues[i];
          }
        }
        else
        {
          previousSearchValues = getSearchValues();
        }
        if (doTimerSearch)
        {
          if (!engineWrapper.SearchInProgress)
          {
            search();
            doTimerSearch = false;
          }
        }
      }
    }

    private void liveSearchCheckBox_CheckedChanged(object sender, EventArgs e)
    {
      searchTimer.Enabled = liveSearchCheckBox.Checked;
    }
  }
}
