﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Security.Authentication;
using System.Text;
using AddressValidation.DataLayer;
using AddressValidation.DataLayer.Entities;
using dataladder.Data;
using dataladder.Licensing;
using DataMatch.Project.Descriptors;
using DataMatch.Project.Helpers;
using Samples.AddressValidation.BL.Matching;
using Samples.AddressValidation.BL.Settings;
using Samples.AddressValidation.BL.Tables;
using Samples.AddressValidation.BL.Transformation;

namespace Samples.AddressValidation.BL
{
    /// <summary>
    /// Class that manages storages, matching and transformations.
    /// This is singlton class. For getting instance call 'Current' property.
    /// 
    /// It is important in the beginning of using call method Init().
    /// 
    /// </summary>
    public class AddressCleaner
    {
        private static AddressCleaner _instance;

        private GlobalSettings _settings { get; set; }
        private MatchingSettings _matchingSettings { get; set; }
        /// <summary>
        /// Table with previous user requests
        /// </summary>
        public DataTable HoldingTable => _historyTable;
        /// <summary>
        /// Table with duplicates from SQL table
        /// </summary>
        public DataTable SuppressedTable => _suppressedTable;
        /// <summary>
        /// Nable with uniquerecords from SQL table
        /// </summary>
        public DataTable MasterTable => _masterDataTable;
        /// <summary>
        /// list of finished steps
        /// </summary>
        public String CompletedStages { get; private set; }
        /// <summary>
        /// Contains information about results of last user request
        /// </summary>
        public Boolean? IsLastInsertedRecordUnique { get; private set; } = null;

        private readonly RegistrationWrapper _registration;
        private readonly DateTime _registrationExpirationDate;

        private readonly DefinitionBuilder _definitionBuilder;
        private readonly AddressMatcher _addressMatcher;
        private readonly MasterTable _masterTable;
        private readonly BufferTable _bufferTable;
        private readonly HoldingTable _holdingTable;
        private readonly Cleaner _recordCleaner;

        private String[] _fieldNames;

        private DataTable _historyTable;
        private DataTable _suppressedTable;
        private DataTable _masterDataTable;
               
        private AddressCleaner()
        {
            // Load settings from app.config or web.config
            _settings = new GlobalSettings();
            _settings.Load();

            // create, clean, delete folders
            InitDirectories();

            _registration = new RegistrationWrapper();
            _registration.CustomPathForRegistrationFile = _settings.LicensePath;
            _registrationExpirationDate = RegistrationWrapper.ExpirationDate;

            _definitionBuilder = new DefinitionBuilder();
            
            _matchingSettings = new MatchingSettings(_settings.BaseDirectory, _settings.TempPath, _settings.WorkingSubfolder);
            _matchingSettings.MatchDefinitionManager = _definitionBuilder.DefinitionsManager;

            _addressMatcher = new AddressMatcher(_matchingSettings);
            // create three tables:
            // 'master' - main table that contains records from data source
            // 'buffer' and 'holding' - auxiliary tables
            // 'buffer' contains cleaned user input
            // 'holding' peek up records that was added in SQL server during working of this application
            _masterTable = new MasterTable(_settings.DataPath, _settings.ConnectionString, _settings.DbTableName);
            _bufferTable = new BufferTable(_settings.DataPath);
            _holdingTable = new HoldingTable(_settings.DataPath);

            // create transformation settings
            if (_settings.UseProjects)
            {
                //settings from project file
                ProjectSpec projectSpec = new ProjectSpec();
                ProjectSpecHelper.LoadProject(projectSpec, _settings.TransformationProjectPath);
                var sources = projectSpec.DataSource.DataSources;

                _recordCleaner = new RecordCleanerByProject(_settings.TempPath, _settings.AddressDbPath, _settings.GeoDbPath,
                    sources[0].StandardizedInfo, sources[0].MergingInfo,
                    _bufferTable.InputTable, _bufferTable.CleanedTable);
            }
            else
            {
                //fixed hardcoded settings
                _recordCleaner = new RecordCleaner(_settings.AddressDbPath, _settings.GeoDbPath, 
                    _bufferTable.InputTable, _bufferTable.CleanedTable);
            }
        }

        /// <summary>
        /// Get or create instance of prepared engine wrapper
        /// </summary>
        public static AddressCleaner Current
        {
            get
            {
                if (_instance == null)
                {
                    _instance = new AddressCleaner();
                }

                return _instance;
            }
        }

        /// <summary>
        /// Initializing
        /// </summary>
        public void Init()
        {
            InitRegistration();
            InitDatabase();
            InitCommonNames();
            InitTables();
            InitFieldNames();
            InitGuiTables();
            InitRecordCleaner();
            InitDefinitionBuilder();
            InitAddressMatcher();
        }
       
        /// <summary>
        /// Clean user input, found for them the duplicates from master table, 
        /// inserting in source SQL table with flag about existing duplicates
        /// </summary>
        /// <param name="values"></param>
        public void AnalyzeRecord(String[] values)
        {
            InsertRecords(values, _fieldNames,
                out _historyTable,
                out DataTable transTable, out DataTable match,
                out _masterDataTable, out _suppressedTable,
                out String[] msgs);

            StringBuilder sb = new StringBuilder();
            Array.ForEach(msgs, msg => sb.AppendLine(msg + "<br/>"));
            CompletedStages = sb.ToString();
        }

        public String Analyze(String firstName, String lastName, String address1, String zip)
        {
            _recordCleaner.ClearTables();

            _bufferTable.InputTable.SetData(firstName, 0, "FirstName");
            _bufferTable.InputTable.SetData(lastName, 0, "LastName");
            _bufferTable.InputTable.SetData(address1, 0, "Address1");
            _bufferTable.InputTable.SetData(zip, 0, "Zip");

            _recordCleaner.Process();

            Boolean hasDups = _addressMatcher.FindDuplicates();

            return hasDups ? "Duplicate" : "Unique";
        }

        /// <summary>
        /// Clean user input, found for them the duplicates from master table, 
        /// inserting in source SQL table with flag about existing duplicates
        /// </summary>
        /// <param name="values">user input</param>
        /// <param name="colnames">name of columns</param>
        /// <param name="history">table with previous user inputs</param>
        /// <param name="transTable">cleaned user input</param>
        /// <param name="match">table with duplicates</param>
        /// <param name="master">unique records from SQL table</param>
        /// <param name="dupl">duplicate records from SQL table</param>
        /// <param name="msgs">Array of string for showing to user about results</param>
        /// <param name="reverseOrderResultTable">should this function return big tables in reverse order</param>
        public void InsertRecords(
            String[] values, String[] colnames,
            out DataTable history, out DataTable transTable, out DataTable match,
            out DataTable master, out DataTable dupl,
            out String[] msgs, Boolean reverseOrderResultTable = true)
        {            
            msgs = null;
            history = _historyTable;
            transTable = null;
            match = null;
            master = null;
            dupl = null;

            List<String> messages = new List<String>();

            //Check correctness input data
            if (values == null || colnames == null)
                throw new ArgumentException();
            if (values.Length != colnames.Length)
                throw new ArgumentException();
            
            //save user input
            AddToHistoryTable(colnames, values);

            //fill buffer and standardize it - cleaning user input
            _recordCleaner.ClearTables();
            String[] cols = _bufferTable.InputTable.GetColumnNames();
            for (Int32 i = 0; i < colnames.Length; i++)
            {
                if (cols.Contains(colnames[i]))
                {
                    _bufferTable.InputTable.SetData(values[i], 0, colnames[i]);
                }
            }
            _recordCleaner.Process();
            messages.Add($"Input record transformed");
            
            transTable = ConvertOnDriveToDataTable(_recordCleaner.CleanedTable);
            //try to define successfullity of address verification
            Boolean verifiedSuccesful = false;
            if (transTable.Rows.Count > 0)
            {
                if (transTable.Columns.Contains("V Status"))
                {
                    if (transTable.Rows[0]["V Status"] is String status)
                    {
                        if (status.Equals("V"))
                            verifiedSuccesful = true;
                    }
                }
            }
            if (verifiedSuccesful)
            {
                messages.Add($"The address verified successfully");
            }
            else
            {
                messages.Add($"The address not verified");
            }

            //search duplicates
            Boolean hasDups = _addressMatcher.FindDuplicates();
            if (hasDups)
            {
                WriteToSuppresedTable(transTable, verifiedSuccesful);
                IsLastInsertedRecordUnique = false;
                messages.Add($"Duplicates found");
                messages.Add($"Input record added into suppressed table");
            }
            else
            {
                _holdingTable.InsertRecord(colnames, values);
                _addressMatcher.RefreshHoldingTable();
                WriteToMasterTable(transTable, verifiedSuccesful);
                IsLastInsertedRecordUnique = true;
                messages.Add($"Input record does not have duplicates");
                messages.Add($"Input record added into the master table");
            }

            master = GetMasterTable(reverseOrderResultTable); 
            dupl = GetSuppressedTable(reverseOrderResultTable);
            msgs = messages.ToArray();
        }

        /// <summary>
        /// reads from SQL server all records with flag 'duplicate'
        /// </summary>
        /// <param name="orderBack">if true then fresh records will be in the beginning of the result table</param>
        public DataTable GetSuppressedTable(Boolean orderBack = true)
        {
            var list = ContactDbManager.GetContacts(true);
            return ContactInfoHelper.ListContactInfoToDataTable(list, orderBack);
        }

        /// <summary>
        /// reads from SQL server all records with flag 'unique'
        /// </summary>
        /// <param name="orderBack">if true then fresh records will be in the beginning of the result table</param>
        /// <returns></returns>
        public DataTable GetMasterTable(Boolean orderBack = true)
        {
            var list = ContactDbManager.GetContacts(false);
            return ContactInfoHelper.ListContactInfoToDataTable(list, orderBack);
        }
        
        /// <summary>
        /// just getter
        /// </summary>
        /// <returns></returns>
        public DataTable GetHistoryTable()
        {
            return _historyTable;
        }

        /// <summary>
        /// Deletes all changes in SQL server.
        /// Reloads this application
        /// </summary>
        public void Reset()
        {
            ContactDbManager.InitMasterTable();
            _historyTable = null;

            _recordCleaner.ClearTables();

            Close();

            _instance = null;
            Current.Init();
        }

        public void Close()
        {
            _addressMatcher.Dispose();
            _definitionBuilder.Dispose();
            _holdingTable.Dispose();
            _bufferTable.Dispose();
            _masterTable.Dispose();
        }

        /// <summary>
        /// Creates, cleans folders for this application
        /// </summary>
        private void InitDirectories()
        {
            if (System.IO.Directory.Exists(_settings.DataPath))
            {
                System.IO.Directory.Delete(_settings.DataPath, true);
            }

            dataladder.IO.IOHelper.CreateDirectoryIfNotExist(_settings.WorkingSubfolder);
            dataladder.IO.IOHelper.CreateDirectoryIfNotExist(_settings.DataPath);
            dataladder.IO.IOHelper.CreateDirectoryIfNotExist(_settings.TempPath);
        }

        /// <summary>
        /// Writes to SQL server one record with flag 'duplicate'
        /// </summary>
        /// <param name="dt">DataTable with one row - record that we want to write</param>
        /// <param name="isVerified">was address verified for this record?</param>
        private void WriteToSuppresedTable(DataTable dt, Boolean isVerified)
        {
            var ci = ContactInfoHelper.CreateContactInfo(dt, true, isVerified);
            ContactDbManager.AddContact(ci);
        }

        /// <summary>
        /// Writes to SQL server one record with flag 'master'
        /// </summary>
        /// <param name="dt">DataTable with one row - record that we want to write</param>
        /// <param name="isVerified">was address verified for this record?</param>
        private void WriteToMasterTable(DataTable dt, Boolean isVerified)
        {
            var ci = ContactInfoHelper.CreateContactInfo(dt, false, isVerified);
            ContactDbManager.AddContact(ci);
        }

        /// <summary>
        /// Creates the history table - table for keeping user requests
        /// </summary>
        /// <param name="colNames"></param>
        private void InitHistoryTable(String[] colNames)
        {
            _historyTable = new DataTable("HoldingTable");
            for (Int32 i = 0; i < colNames.Length; i++)
            {
                _historyTable.Columns.Add(colNames[i], typeof(String));
            }
        }

        /// <summary>
        /// Adds in the history table another one user request
        /// </summary>
        /// <param name="colnames"></param>
        /// <param name="values"></param>
        private void AddToHistoryTable(String[] colnames, String[] values)
        {
            if (_historyTable == null)
            {
                InitHistoryTable(colnames);
            }

            DataRow row = _historyTable.NewRow();
            for (Int32 i = 0; i < colnames.Length; i++)
            {
                if (_historyTable.Columns.Contains(colnames[i]))
                {
                    row[colnames[i]] = values[i];
                }
            }
            _historyTable.Rows.Add(row);
            _historyTable.AcceptChanges();
        }

        /// <summary>
        /// Checks Dataladder registration
        /// </summary>
        private void InitRegistration()
        {
            //_registration.CustomPathForRegistrationFile = _settings.LicensePath;
            DateTime expirationDate = RegistrationWrapper.ExpirationDate;

            if (expirationDate < DateTime.Now)
            {
                throw new AuthenticationException("License expired!");
            }
        }

        /// <summary>
        /// Loads Dictionaries for checking First Name
        /// </summary>
        private void InitCommonNames()
        {
            dataladder.Data.Format.CommonNames.LoadDictionary();
            dataladder.Data.Format.Gender.LoadDictionary();
        }
        /// <summary>
        /// Checks SQL server master table. If it is not exist then create it
        /// </summary>
        private void InitDatabase()
        {
            ContactDbManager.InitDatabase();
        }

        /// <summary>
        /// Creates tables with which this application works.
        /// Defines their structure.
        /// Loads records from master SQL server table
        /// </summary>
        private void InitTables()
        {
            _masterTable.Init();
            _bufferTable.Init(_masterTable.Table);
            _holdingTable.Init(_masterTable.Table);
        }

        /// <summary>
        /// Define column names
        /// </summary>
        private void InitFieldNames()
        {
           _fieldNames = new String[]
           {
                nameof(ContactInfo.FirstName),
                nameof(ContactInfo.LastName),
                nameof(ContactInfo.Address1),
                nameof(ContactInfo.Address2),
                nameof(ContactInfo.City),
                nameof(ContactInfo.State),
                nameof(ContactInfo.Zip),
                nameof(ContactInfo.Phone),
                nameof(ContactInfo.Email)
            };
        }

        /// <summary>
        /// Reload table for GUI applications
        /// </summary>
        private void InitGuiTables()
        {
            InitHistoryTable(_fieldNames);
            _suppressedTable = GetSuppressedTable(true);
            _masterDataTable = GetMasterTable(true);
        }

        /// <summary>
        /// Create transformation diagram
        /// </summary>
        private void InitRecordCleaner()
        {
            _recordCleaner.CreateDiagram(_bufferTable.InputTable, _bufferTable.CleanedTable);
        }
        
        /// <summary>
        /// Creates matching settings
        /// </summary>
        private void InitDefinitionBuilder()
        {
            if (_settings.UseProjects)
            {
                // Load settings from project
                _definitionBuilder.Init(_masterTable.Table, _bufferTable.CleanedTable, _holdingTable.Table,
                    _settings.MatchingProjectPath);
            }
            else
            {
                // settings are hardcoded and fixed
                _definitionBuilder.Init(_masterTable.Table, _bufferTable.CleanedTable, _holdingTable.Table, null);
            }
        }

        /// <summary>
        /// Creates match engine. Makes first indexing of tables.
        /// Preparing for matching
        /// </summary>
        private void InitAddressMatcher()
        {
            _matchingSettings.MasterOnDrive = _masterTable.Table;
            _matchingSettings.BufferOnDrive = _bufferTable.CleanedTable;
            _matchingSettings.HoldingOnDrive = _holdingTable.Table;
            _addressMatcher.Init();
        }

        /// <summary>
        /// Converts data from OnDriveTable type to DataTable type
        /// </summary>
        /// <param name="drive">OnDriveTable instance</param>
        /// <returns></returns>
        private DataTable ConvertOnDriveToDataTable(OnDriveTable drive)
        {
            if (drive == null)
                return null;

            DataTable dt = new DataTable(drive.Name);
            for (Int32 i = 0; i < drive.ColumnCount; i++)
            {
                dt.Columns.Add(drive.GetColumnName(i), typeof(String));
            }
            for (Int32 i = 0; i < drive.RecordCount; i++)
            {
                DataRow row = dt.NewRow();
                for (Int32 j = 0; j < drive.ColumnCount; j++)
                {
                    row[j] = drive.GetData(i, j);
                }
                dt.Rows.Add(row);
            }
            return dt;
        }
    }
}
