﻿using System;
using System.Data;
using dataladder.Data;
using dataladder.Matching;
using dataladder.XtraGridHelper;
using Samples.AddressValidation.BL.Settings;

namespace Samples.AddressValidation.BL.Matching
{
    /*  This class is a wrapper for Match Engine
     *  
     *  After creating it is important to call Init() 
     *  This function contains the hardcoded configuration of 
     *  Matching Pairs and input data sources
     * */
    public class AddressMatcher : IDisposable
    {
        private MatchingSettings _settings;
        private MatchEngine _matchEngine;
        
        /// <summary>
        /// Creates instance.
        /// </summary>
        /// <param name="settings">rules for matching</param>
        public AddressMatcher(MatchingSettings settings)
        {
            _settings = settings;
            _matchEngine = new MatchEngine(_settings.MatchDefinitionManager, _settings.AllRecordsInGoupMustBeSimilar,
                _settings.TempPath, _settings.DataPath, _settings.Name);
        }

        /// <summary>
        /// Defines matching pairs. Prepares for matcing. Prepares indexes. 
        /// </summary>
        public void Init()
        {
            _matchEngine.InputDataMapperDict.Clear();
            // No. 0 data source
            _matchEngine.InputDataMapperDict.Add(_settings.MasterOnDrive.Name, _settings.MasterOnDrive);
            // No. 1 data source
            _matchEngine.InputDataMapperDict.Add(_settings.BufferOnDrive.Name, _settings.BufferOnDrive);
            // No. 2 data source
            _matchEngine.InputDataMapperDict.Add(_settings.HoldingOnDrive.Name, _settings.HoldingOnDrive);

            _settings.MasterOnDrive.PriorityForMemory = OnDriveTable.PrioritiesForMemory.Normal;
            _settings.BufferOnDrive.PriorityForMemory = OnDriveTable.PrioritiesForMemory.Normal;
            _settings.HoldingOnDrive.PriorityForMemory = OnDriveTable.PrioritiesForMemory.Normal;

            // define which sources will be matched with each other
            // master - user input
            // user input - previous insertings
            _matchEngine.DataSourceIndexPairList.Clear();
            _matchEngine.DataSourceIndexPairList.Add(new MatchEngine.DataSourceIndexPair(0, 1));
            _matchEngine.DataSourceIndexPairList.Add(new MatchEngine.DataSourceIndexPair(2, 1));

            // prepare the match engine
            _matchEngine.DoIndex();

            // Creates a list of sources those can be cached
            // We work with the master table that doesn't change, so we can cache this data source
            // it is No. 0
            // Others source change, so they are not included in this list
            var list = new System.Collections.Generic.List<int>() { 0 };
            _matchEngine.SetCachedDataSources(list);
        }
        
        /// <summary>
        /// Recalculate indexes for table with previous insertings
        /// This is data source No.2 - holding table
        /// </summary>
        internal void RefreshHoldingTable()
        {
            _matchEngine.ReindexSingleDataSource(2);
        }

        /// <summary>
        /// Search duplicates
        /// </summary>
        /// <returns></returns>
        public Boolean FindDuplicates()
        {
            // The source No.1 always changes (because this is user input) so refresh indexes for it
            _matchEngine.ReindexSingleDataSource(1);

            _matchEngine.LoadAllInMemory = true;

            // Matching and postprocessing
            _matchEngine.DoMatch(_settings.MaxMatchesForOneRow, _settings.ClearAllAfterMatching, _settings.MaxMatchesPerGroup);
            _matchEngine.ProcessFinalResults(_settings.ClearAllAfterMatching);

            // PairsScoresTable contains all founded duplicates 
            return _matchEngine.PairsScoresTable.RecordCount > 0;
        }

        public void Dispose()
        {
            _matchEngine?.Dispose();
        }

    }
}
