﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using dataladder.Data;
using dataladder.Matching;
using DataMatch.Project.Descriptors;
using DataMatch.Project.Helpers;
using log4net.Core;
using Samples.Common.Builders;

namespace Samples.AddressValidation.BL.Matching
{
    /* This class is needed for defining match rules. 
     * These rules are kept in 'DefinitionManager'  that connects criteria, 
     * definitions, and columns of data sources 
     * also it connects columns from different data sources with each other
     * 
     * If during calling Init() method a project file is 
     * chosen then setting will be loaded from the project. 
     * If a project file is not chosen then rules are hardcoded in MakeStub() method
     * 
     * */
    public class DefinitionBuilder : IDisposable
    {
        public MultipleMatchDefinitionsManager DefinitionsManager => _definitionsManager;

        private MultipleMatchDefinitionsManager _definitionsManager;
        private OnDriveTable _masterTable;
        private OnDriveTable _holdingTable;
        private OnDriveTable _bufferTable;

        public DefinitionBuilder()
        {
            _definitionsManager = new MultipleMatchDefinitionsManager();
            _definitionsManager.AvailableFields = new AvailableFields();
        }

        /// <summary>
        /// Creates matching rules. If project file not choosed then fixed hardcoded settings will be created
        /// </summary>
        /// <param name="masterTable">storage with data from SQL table</param>
        /// <param name="bufferTable">cleaned user input</param>
        /// <param name="holdingTable">storage with previous user requests</param>
        /// <param name="projectPath">path to project file</param>
        public void Init(OnDriveTable masterTable, OnDriveTable bufferTable, OnDriveTable holdingTable, 
            String projectPath = null)
        {
            if (System.IO.File.Exists(projectPath))
            {
                LoadFromProject(projectPath, masterTable, bufferTable, holdingTable);
            }
            else
            {
                MakeStub(masterTable, bufferTable, holdingTable);
            }
        }
        
        /// <summary>
        /// Loads matching settings from project file. Attaches these rules to three onDriveTable.
        /// All three tables must contain same columns that are defined in the project file
        /// </summary>
        /// <param name="projectPath">path to project file</param>
        /// <param name="masterTable">first table it is better if project will be based on this data source</param>
        /// <param name="bufferTable">second table that contain user input</param>
        /// <param name="holdingTable">third table where previous insertings are kept</param>
        private void LoadFromProject(String projectPath, 
            OnDriveTable masterTable, OnDriveTable bufferTable, 
            OnDriveTable holdingTable)
        {
            _masterTable = masterTable;
            _bufferTable = bufferTable;
            _holdingTable = holdingTable;

            // loads settings from project file
            // and attaches them to master table
            ProjectSpec projectSpec = new ProjectSpec();
            ProjectSpecHelper.LoadProject(projectSpec, projectPath);
            var builder = new DefinitionManagerBuilder(projectSpec.MatchDefinition, 
                new List<ITable2CoordsMapper> { _masterTable});
            builder.Build(_definitionsManager);

            // attaches criteria to buffer and holding table
            for (int i = 0; i < _definitionsManager.Count; i++)
            {
                MatchCriteriaList criterias = _definitionsManager[i];
                for (int j = 0; j < criterias.Count; j++)
                {
                    MatchCriteria criteria = criterias[j];
                    bool found;
                    string fieldName = criteria.GetMappedFieldName(_masterTable.Name, out found);
                    if (found)
                    {
                        criteria.MapField(_bufferTable.Name, fieldName);
                        criteria.MapField(_holdingTable.Name, fieldName);
                    }
                }
            }

            // define how fields in different tables map each to other
            AvailableFields avFields = _definitionsManager.AvailableFields;
            AvailableFieldsFromOneTable masterAFOT = avFields[0];
            AvailableFieldsFromOneTable bufferAFOT = new AvailableFieldsFromOneTable();
            bufferAFOT.Table = bufferTable;
            AvailableFieldsFromOneTable holdingAFOT = new AvailableFieldsFromOneTable();
            holdingAFOT.Table = holdingTable;
            avFields.TableList.Add(bufferAFOT);
            avFields.TableList.Add(holdingAFOT);
            var bufferCols = _bufferTable.GetColumnNames();
            var holdingCols = _holdingTable.GetColumnNames();
            int bufferIndex = 1;
            int holdingIndex = 2;
            foreach (MappedFieldsRow fields in avFields.MappedFieldsRowList)
            {
                FieldMapInfo masterFmi = fields?.FirstNotEmptyFieldMapInfo;
                if (masterFmi == null) continue;
                string columnName = masterFmi.FieldName;
#if DEBUG
                string firstTableName = masterFmi.TableName;
                if (!firstTableName.Equals(_masterTable.Name))
                    throw new ArgumentException("Something go wrong !!! AvailableFields contains wrong table Name");
#endif
                if (bufferCols.Contains(columnName))
                {
                    FieldMapInfo fmi = new FieldMapInfo(bufferIndex);
                    fmi.FieldName = columnName;
                    fmi.FieldIndex = _bufferTable[columnName].Index;
                    fmi.TableName = _bufferTable.Name;
                    fmi.Mapped = true;

                    bufferAFOT.Add(fmi);
                    fields.Add(fmi);
                }

                if (holdingCols.Contains(columnName))
                {
                    FieldMapInfo fmi = new FieldMapInfo(holdingIndex);
                    fmi.FieldName = columnName;
                    fmi.FieldIndex = _holdingTable[columnName].Index;
                    fmi.TableName = _holdingTable.Name;
                    fmi.Mapped = true;

                    holdingAFOT.Add(fmi);
                    fields.Add(fmi);
                }
            }
        }
        
        /// <summary>
        /// Creates fixed hardcoded matching settings
        /// </summary>
        /// <param name="masterTable">storage for master data</param>
        /// <param name="bufferTable">storage for an user request</param>
        /// <param name="holdingTable">storage with previous user requests</param>
        private void MakeStub(OnDriveTable masterTable, OnDriveTable bufferTable, OnDriveTable holdingTable)
        {
            _masterTable = masterTable;
            _bufferTable = bufferTable;
            _holdingTable = holdingTable;

            var availableFields = new[] {
                new {
                    Name = _masterTable.Name,
                    Table = _masterTable,
                    Index = 0,
                    Fields = new[] {
                        new {Name = "FirstName", Index = 1},
                        new {Name = "LastName", Index = 2},
                        new {Name = "Address1", Index = 3},
                        new {Name = "Zip", Index = 7}
                    }
                },
                new {
                    Name = _bufferTable.Name,
                    Table = _bufferTable,
                    Index = 1,
                    Fields = new[] {
                        new {Name = "FirstName", Index = 1},
                        new {Name = "LastName", Index = 4},
                        new {Name = "Address1", Index = 5},
                        new {Name = "Zip", Index = 19}
                    }
                },
                new {
                    Name = _holdingTable.Name,
                    Table = _holdingTable,
                    Index = 0,
                    Fields = new[] {
                        new {Name = "FirstName", Index = 1},
                        new {Name = "LastName", Index = 2},
                        new {Name = "Address1", Index = 3},
                        new {Name = "Zip", Index = 7}
                    }
                },

            };

            var fieldMapping = new[]
            {
                new[] {
                    new {TableName = _masterTable.Name, FieldName = "FirstName"},
                    new {TableName = _bufferTable.Name, FieldName = "FirstName"},
                    new {TableName = _holdingTable.Name, FieldName = "FirstName"},
                },
                new[] {
                    new {TableName = _masterTable.Name, FieldName = "LastName"},
                    new {TableName = _bufferTable.Name, FieldName = "LastName"},
                    new {TableName = _holdingTable.Name, FieldName = "LastName"},
                },
                new[] {
                    new {TableName = _masterTable.Name, FieldName = "Address1"},
                    new {TableName = _bufferTable.Name, FieldName = "Address1"},
                    new {TableName = _holdingTable.Name, FieldName = "Address1"},
                },
                new[] {
                    new {TableName = _masterTable.Name, FieldName = "Zip"},
                    new {TableName = _bufferTable.Name, FieldName = "Zip"},
                    new {TableName = _holdingTable.Name, FieldName = "Zip"},
                }
            };

            var criteriaMapping = new[]
            {

                new { Level = 0.9, Mapping = new[] {
                        new {TableName = _masterTable.Name, FieldName = "FirstName"},
                        new {TableName = _bufferTable.Name, FieldName = "FirstName"},
                        new {TableName = _holdingTable.Name, FieldName = "FirstName"},
                    }
                },
                new { Level = 0.9, Mapping = new[] {
                        new {TableName = _masterTable.Name, FieldName = "LastName"},
                        new {TableName = _bufferTable.Name, FieldName = "LastName"},
                        new {TableName = _holdingTable.Name, FieldName = "LastName"},
                    }
                },
                new { Level = 0.9, Mapping = new[] {
                        new {TableName = _masterTable.Name, FieldName = "Address1"},
                        new {TableName = _bufferTable.Name, FieldName = "Address1"},
                        new {TableName = _holdingTable.Name, FieldName = "Address1"},
                    }
                },
                new { Level = 0.9, Mapping = new[] {
                        new {TableName = _masterTable.Name, FieldName = "Zip"},
                        new {TableName = _bufferTable.Name, FieldName = "Zip"},
                        new {TableName = _holdingTable.Name, FieldName = "Zip"},
                    }
                }
            };

            Dictionary<String, Dictionary<String, FieldMapInfo>> fieldInfoMapping = new Dictionary<String, Dictionary<String, FieldMapInfo>>();

            foreach (var table in availableFields)
            {
                AvailableFieldsFromOneTable affot = new AvailableFieldsFromOneTable { Table = table.Table };
                Dictionary<String, FieldMapInfo> tableFieldInfos = new Dictionary<String, FieldMapInfo>();
                fieldInfoMapping.Add(table.Name, tableFieldInfos);

                foreach (var field in table.Fields)
                {
                    FieldMapInfo fmi = new FieldMapInfo(table.Index) {
                        FieldName = field.Name,
                        TableName = table.Name,
                        ColumnTransformation = null,
                        FieldIndex = field.Index
                    };
                    affot.Add(fmi);
                    tableFieldInfos.Add(field.Name, fmi);

                }
                _definitionsManager.AvailableFields.TableList.Add(affot);

            }

            foreach (var mapping in fieldMapping)
            {
                MappedFieldsRow mfr = new MappedFieldsRow();

                foreach (var entry in mapping)
                {
                    FieldMapInfo fmi = fieldInfoMapping[entry.TableName][entry.FieldName];
                    mfr[entry.TableName] = fmi;
                }
                _definitionsManager.AvailableFields.MappedFieldsRowList.Add(mfr);
            }

            _definitionsManager.SetAbsoluteIndices();

            Int32 matchDefinitionIndex = 0;
            MatchCriteriaList matchCriteriaList = new MatchCriteriaList(matchDefinitionIndex);

            foreach (var criteria in criteriaMapping)
            {
                MatchCriteria matchCriteria = CreateFuzzyCriteria(criteria.Level);
                foreach (var cfm in criteria.Mapping)
                {
                    matchCriteria.MapField(cfm.TableName, cfm.FieldName);
                }
                matchCriteriaList.Add(matchCriteria);

            }

            matchCriteriaList.MarkTheFirstFieldInEveryGroup();
            _definitionsManager.Add(matchCriteriaList);
        }

        /// <summary>
        /// Create hardcoded match creteria with specified matching level
        /// </summary>
        /// <param name="level">must be in limits [0.0 ... 1.0]</param>
        /// <returns></returns>
        private MatchCriteria CreateFuzzyCriteria(Double level)
        {
            MatchCriteria fuzzyCriteria = new MatchCriteria
            {
                Fuzzy = true,
                AddWeightToFirstLetter = false,
                Exact = false,
                Numeric = false,
                UseMetaphone = false,
                IgnoreCase = true,
                Level = level,
                GroupLevel = 0.0f,
                MinAllowedLevelInGroup = 0.0f,
                GroupId = -1,
                CrossColumnGroupId = -1,
                Weight = 100.0f,
                MatchingIndex = 0,
                AbsoluteMatchingIndex = 0
            };

            return fuzzyCriteria;
        }

        public void Dispose()
        {
            _definitionsManager.Dispose();
        }
    }
}
