﻿using System;
using System.Linq;
using System.Threading;
using dataladder.Data;
using log4net;

namespace Samples.AddressValidation.BL.Tables
{
    /// <summary>
    /// Wrapper for data that were loaded from SQL server
    /// </summary>
    public class MasterTable : IDisposable
    {
        private static readonly ILog Log = LogManager.GetLogger(typeof(MasterTable));

        /// <summary>
        /// internal storage
        /// </summary>
        public OnDriveTable Table { get; private set; }
        /// <summary>
        /// working folder
        /// </summary>
        public String Path { get; }
        /// <summary>
        /// connection string for source database
        /// </summary>
        public String ConnectionString { get; }
        /// <summary>
        /// table name in data base from which data were loaded
        /// </summary>
        public String DbTableName { get; }

        public const String TableName = "MasterTable";
        /// <summary>
        /// name of the folder for the keeping of intermediate results
        /// </summary>
        private const String UncompleteDirectory = "uncomplete";

        /// <summary>
        /// list of column names that will be saved and processed, other will be ignored
        /// </summary>
        private readonly String[] _fieldsToInclude =
        {
            "Id",
            "FirstName",
            "LastName",
            "Address1",
            "Address2",
            "City",
            "State",
            "Zip",
            "Country",
            "Phone",
            "Email",
            //"Custom1",
            //"Custom2",
            "IsDuplicate",
            "IsVerified"
        };

        /// <summary>
        /// Create internal storage of this wrapper
        /// </summary>
        /// <param name="path">working folder</param>
        /// <param name="connectionString">connection string to database, from which data loaded</param>
        /// <param name="dbTableName">table name from database</param>
        public MasterTable(String path, String connectionString, String dbTableName)
        {
            Path = path;
            ConnectionString = connectionString;
            DbTableName = dbTableName;
            Table = new OnDriveTable(Path, TableName);
        }

        /// <summary>
        /// Loads from database the data
        /// </summary>
        /// <returns></returns>
        public Boolean Init()
        {
            Boolean result = false;

            try
            {
                //Read data source
                IReader reader = dataladder.Data.IReaderHelper.GetReader(SourceDb.SqlServer, ConnectionString, String.Empty, DbTableName);
                ReaderConfiguration configuration = reader.GetConfiguration();
                configuration.TableName = DbTableName;
                configuration.OriginalTableName = DbTableName;
                reader.SetConfiguration(configuration);
                reader.ReadTable(configuration, toDetermineFields: true);

                for (Int32 fieldIndex = 0; fieldIndex < reader.Fields.RecordCount; fieldIndex++)
                {
                    Field field = reader.Fields[fieldIndex];
                    field.Included = _fieldsToInclude.FirstOrDefault(fn => fn.Equals(field.OriginalName, StringComparison.OrdinalIgnoreCase)) != null;
                }

                String uncompletePath = System.IO.Path.Combine(Path, UncompleteDirectory);

                var inputTable = ReaderToVariableTableConvertor.Copy(reader, uncompletePath, configuration.OriginalTableName, out String error,
                    new CancellationToken(), fileNameBase: TableName);

                reader.Disconnect();
                //clean intermediate storage
                Table.ToDeleteFilesAfterClosing = true;
                Table.Dispose();

                inputTable?.Move(Path);

                Table = inputTable;

                result = String.IsNullOrEmpty(error);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }

            return result;
        }

        public void Dispose()
        {
            Table.ToDeleteFilesAfterClosing = true;
            Table.Dispose();
        }
    }
}
