﻿using dataladder.Data;
using dataladder.Data.DataTransformation;
using Samples.Common.Builders;
using System;

namespace Samples.AddressValidation.BL.Transformation
{
    /*
     *  Class for defining the interface for classes that make cleansing 
     *  and standardization.
     *  
     *  There are two derived classes: the first works with settings from a project, 
     *  another works with fixed settings
     * */
    public abstract class Cleaner
    {
        /// <summary>
        /// Storage for keeping a raw user request
        /// </summary>
        public OnDriveTable InputTable => _inputTable;
        protected OnDriveTable _inputTable;
        /// <summary>
        /// Storage for keeping cleansed request
        /// </summary>
        public OnDriveTable CleanedTable => _cleanedTable;
        protected OnDriveTable _cleanedTable;

        protected TransformationDiagram _transformationDiagram;

        /// <summary>
        /// Cleans buffer storage for user request
        /// </summary>
        public void ClearTables()
        {
            ClearTable(_inputTable);
        }

        /// <summary>
        /// Cleans storage
        /// </summary>
        /// <param name="table">table that will be cleansed</param>
        private void ClearTable(OnDriveTable table)
        {
            for (Int32 i = 0; i < table.ColumnCount; i++)
            {
                table.SetData(null, 0, i);
            }
        }
        /// <summary>
        /// Cleans data
        /// </summary>
        public void Process()
        {
            Int32 rowCount = _inputTable.RecordCount;

            _cleanedTable.MakeWritable();

            for (Int32 rowIndex = 0; rowIndex < rowCount; rowIndex++)
            {
                _transformationDiagram.Process(rowIndex);

                for (Int32 i = 0; i < _transformationDiagram.Outputs.Count; i++)
                {
                    String colName = _transformationDiagram.Outputs[i].Name;
                    Object value = _transformationDiagram.Outputs[i].Value;
                    _cleanedTable.SetData(value, rowIndex, colName);
                }
            }
        }

        /// <summary>
        /// Creates standardization rules and attaches them to input storage
        /// </summary>
        /// <param name="inputTable">storage where will be loaded user inputs</param>
        /// <param name="cleanedTable">strorage where results of cleansing will be kept</param>
        public abstract void CreateDiagram(OnDriveTable inputTable, OnDriveTable cleanedTable);

        /// <summary>
        /// Method for internal purposes. This method is needed 
        /// for correct work DataLadder libraries
        /// </summary>
        protected Object OnGetInputData(Int32 inputIndex, Int32 taskIndex, Int32 rowIndex)
        {
            return _inputTable.GetData(rowIndex, inputIndex);
        }
    }
}
