﻿using System;
using dataladder.AddressVerification;
using dataladder.Data;
using dataladder.Data.DataTransformation;
using DataMatch.AddressVerification.Cass;

namespace Samples.AddressValidation.BL.Transformation
{
    /*  The class that makes cleansing and standardization by 
     *  hardcoded settings
     * */
    public class RecordCleaner: Cleaner
    {
        //Parser for CASS verification
        // Required for address transformation.
        private CassParser _cassParser;

        /// <summary>
        /// pathes to CASS folders
        /// </summary>
        private String _cassPath, _geoPath;

        /// <summary>
        /// Create instance
        /// </summary>
        /// <param name="cassPath">path to CASS folder</param>
        /// <param name="geoPath">path to CASS Geocoder folder</param>
        /// <param name="inputTable">Storage with input data</param>
        /// <param name="outputTable">Storage for results</param>
        public RecordCleaner(String cassPath, String geoPath, OnDriveTable inputTable, OnDriveTable cleanedTable)
        {
            _cassPath = cassPath;
            _geoPath = geoPath;

            _inputTable = inputTable;
            _cleanedTable = cleanedTable;
            _cassParser = new CassParser();

            _transformationDiagram = new TransformationDiagram(0);
            _transformationDiagram.OnGetInputData += OnGetInputData;

            InitCass();
        }

        /// <summary>
        /// Loads libraries for address verification
        /// </summary>
        private void InitCass()
        {
            _cassParser.InitCassIfNeeded(_cassPath, _geoPath);
        }

        /// <summary>
        /// Creates standardization rules and attaches them to input storage
        /// </summary>
        /// <param name="inputTable">Storage with raw data</param>
        /// <param name="cleanedTable">Storage for results</param>
        public override void CreateDiagram(OnDriveTable inputTable, OnDriveTable cleanedTable)
        {
            _inputTable = inputTable;

            _transformationDiagram.Clear();

            AddInput("Id", 0);
            // create FirstName transformation block
            FirstNameBlock firstNameBlock = CreateFirstNameBlock();
            //attach column to transformation block
            AddFirstNameBlock(firstNameBlock, "FirstName", 1);
            AddInput("LastName", 2);
            //create address transformation block
            AddressCassBlock cassBlock = CreateCassBlock();
            //attach column to transformation block
            AddCassBlock(cassBlock, "Address1", 3, CassInputTypes.PrimaryAddress);
            AddInput("Address2", 4);
            //attach column to transformation block
            AddCassBlock(cassBlock, "City", 5, CassInputTypes.CityName);
            AddInput("State", 6);
            //attach column to transformation block
            AddCassBlock(cassBlock, "Zip", 7, CassInputTypes.ZipCode);
            AddInput("Country", 8);
            AddInput("Phone", 9);
            AddInput("Email", 10);
            //AddInput("Custom1", 11);
            //AddInput("Custom2", 12);
            AddInput("IsDuplicate", 11);
            AddInput("IsVerified", 12);

            // Prepare diagram
            _transformationDiagram.CreateOutputs();
            _transformationDiagram.Prepare();

            // Add columns to transformed table
            for (Int32 i = 0; i < _transformationDiagram.Outputs.Count; i++)
            {
                String columnName = _transformationDiagram.Outputs[i].Name;
                _cleanedTable.AddField(columnName);
            }

            ClearTables();
        }

        /// <summary>
        /// Creates setting for CASS - which columns will be added during CASS verification
        /// </summary>
        private AddressVerificationSettings CreateAddressVerificationSettings()
        {
            AddressVerificationSettings addressVerificationSettings = new AddressVerificationSettings();

            addressVerificationSettings.OutputColumnSettings.ForEach(s =>
            {
                switch (s.CassOutputPart)
                {
                    case CassOutputParts.Status:
                    case CassOutputParts.CompanyFirm:
                    case CassOutputParts.PrimaryAddress:
                    case CassOutputParts.SecondaryAddress:
                    case CassOutputParts.City:
                    case CassOutputParts.State:
                    case CassOutputParts.ZipCode:
                    case CassOutputParts.PostalCode:
                    case CassOutputParts.BuildingNumber:
                    case CassOutputParts.StreetName:
                        s.Include = true;
                        break;
                    default:
                        s.Include = false;
                        break;
                }
            });

            return addressVerificationSettings;
        }

        /// <summary>
        /// Create CASS block
        /// </summary>
        /// <returns></returns>
        private AddressCassBlock CreateCassBlock()
        {
            AddressVerificationSettings settings = CreateAddressVerificationSettings();
            AddressCassBlock cassBlock = new AddressCassBlock(_cassParser.CassAddress, _cassParser.CassGeoCoder, settings);
            _transformationDiagram.AddTransformationBlock(cassBlock);
            _transformationDiagram.ContainsCassBlock = true;

            return cassBlock;
        }


        private FirstNameBlock CreateFirstNameBlock()
        {
            FirstNameBlock firstNameBlock = new FirstNameBlock();
            _transformationDiagram.AddTransformationBlock(firstNameBlock);

            return firstNameBlock;
        }

        private DataFlow AddInput(String fieldName, Int32 fieldIndex)
        {
            return _transformationDiagram.AddInput(fieldName, fieldIndex);
        }

        /// <summary>
        /// Add to specifed column FirstName transformation block 
        /// </summary>
        /// <param name="firstNameBlock">FirstName block</param>
        /// <param name="fieldName">column name</param>
        /// <param name="fieldIndex">column index in the table</param>
        /// <returns></returns>
        private DataFlow AddFirstNameBlock(FirstNameBlock firstNameBlock, String fieldName, Int32 fieldIndex)
        {
            DataFlow firstNameInput = _transformationDiagram.AddInput(fieldName, fieldIndex);
            firstNameBlock.AddInput(firstNameInput);

            return firstNameInput;
        }

        /// <summary>
        /// Add specified column to CASS verification, attachthis column to CASS input type
        /// </summary>
        /// <param name="cassBlock">AV block</param>
        /// <param name="fieldName">column name</param>
        /// <param name="fieldIndex">column index in the table</param>
        /// <param name="inputType">CASS input type</param>
        /// <returns></returns>
        private DataFlow AddCassBlock(AddressCassBlock cassBlock, String fieldName, Int32 fieldIndex, CassInputTypes inputType)
        {
            DataFlow input = _transformationDiagram.AddInput(fieldName, fieldIndex);
            cassBlock.AddInput(input, inputType);

            return input;
        }


    }
}
