﻿using System;
using dataladder.AddressVerification;
using dataladder.Data;
using dataladder.Data.DataTransformation;
using DataMatch.Project.Descriptors.DataSource;
using Samples.Common.Builders;

namespace Samples.AddressValidation.BL.Transformation
{
    /*  The class that makes cleansing and standardization by 
     *  defined in project file settings
     * */
    public class RecordCleanerByProject : Cleaner
    {
        /// <summary>
        /// Settings for standardization for each column
        /// </summary>
        private ColumnTransformationList _collumnTransform;
        /// <summary>
        /// A merge columns settings
        /// </summary>
        private MergeBlocksStorage _mergeTransform;
        /// <summary>
        /// An address verifaction settings - which columns will be added
        /// during CASS verification
        /// </summary>
        private AddressVerificationSettings _addressVerificationSettings;
        /// <summary>
        /// working paths
        /// </summary>
        private String _tempPath, _cassPath, _geoPath;

        /// <summary>
        /// Create instance
        /// </summary>
        /// <param name="tempPath">path to Temp folder that will be used for temporary results</param>
        /// <param name="cassPath">path to CASS folder</param>
        /// <param name="geoPath">path to CASS Geocoder folder</param>
        /// <param name="standardizedSpec">Specification of standardization rules</param>
        /// <param name="mergingSpec">Specification of merging rules</param>
        /// <param name="inputTable">Storage with input data</param>
        /// <param name="outputTable">Storage for results</param>
        /// <param name="avSettings">Setting for CASS - which columns will be added during CASS verification</param>
        public RecordCleanerByProject(string tempPath, String cassPath, String geoPath, 
            ColumnTransformationsSpec standardizedSpec, MergeInfoSpec mergingSpec, 
            OnDriveTable inputTable, OnDriveTable outputTable,
            AddressVerificationSettings avSettings = null)
        {
            _tempPath = tempPath;
            _cassPath = cassPath;
            _geoPath = geoPath;

            _inputTable = inputTable;
            _cleanedTable = outputTable;

            TransformationBuilder builder = new TransformationBuilder(standardizedSpec, mergingSpec, tempPath);
            _collumnTransform = builder.BuildStandardizedSettings();
            _mergeTransform = builder.BuildMergingSettings();
            _addressVerificationSettings = avSettings;
        }

        /// <summary>
        /// Creates standardization rules and attaches them to input storage
        /// </summary>
        /// <param name="inputTable">Storage with raw data</param>
        /// <param name="cleanedTable">Storage for results</param>
        public override void CreateDiagram(OnDriveTable inputTable, OnDriveTable cleanedTable)
        {
            _inputTable = inputTable;

            var diagramBuilder = new TransformationDiagramBuilder(_tempPath, _cassPath, _geoPath,
                _collumnTransform, _mergeTransform, inputTable, _addressVerificationSettings);
            _transformationDiagram = diagramBuilder.Build();
            _transformationDiagram.OnGetInputData += OnGetInputData;

            // Prepare diagram
            _transformationDiagram.CreateOutputs();
            _transformationDiagram.Prepare();

            // Add columns to transformed table
            for (Int32 i = 0; i < _transformationDiagram.Outputs.Count; i++)
            {
                String columnName = _transformationDiagram.Outputs[i].Name;
                _cleanedTable.AddField(columnName);
            }

            ClearTables();
        }
    }
}
