﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Reflection;
using System.Xml;
using AddressValidation.DataLayer.DataModel;
using AddressValidation.DataLayer.Entities;
using log4net;

namespace AddressValidation.DataLayer
{
    /// <summary>
    /// Class for simplifying access to the database from the application
    /// 
    /// This class contains methods for populating and clensing the table, 
    /// adding and reading records.
    /// </summary>
    public static class ContactDbManager
    {
        private static readonly ILog Log = LogManager.GetLogger(typeof(ContactDbManager));

        /// <summary>
        /// Connection string from configuration file
        /// </summary>
        public static String ConnectionString => Properties.Settings.Default.ContactDataDbConnectionString;

        /// <summary>
        /// This method checks that database defined in configuration file 
        /// exists. If doesn't exist then creates it
        /// </summary>
        public static void InitDatabase()
        {
            SqlConnectionStringBuilder scsb = new SqlConnectionStringBuilder(ConnectionString);
            String dbName = scsb.InitialCatalog;
            scsb.InitialCatalog = "master";
            String connectionString = scsb.ToString();

            if (DatabaseExists(connectionString, dbName))
            {
                return;
            }

            using (AddressDbModelDataContext dataContext = new AddressDbModelDataContext(Properties.Settings.Default.ContactDataDbConnectionString))
            {
                dataContext.CreateDatabase();
            }

            InsertSampleData();
        }
        
        /// <summary>
        /// Clears table in database and fills it with default values
        /// </summary>
        public static void InitMasterTable()
        {
            ClearTable(ConnectionString);
            InsertSampleData();
        }

        /// <summary>
        /// Execute SQL command
        /// </summary>
        /// <param name="connectionString">connection string to database</param>
        /// <param name="sql">text of command</param>
        internal static void ExecuteCommand(String connectionString, String sql)
        {
            using (SqlConnection conn = new SqlConnection(connectionString))
            {
                conn.Open();

                using (SqlCommand cmd = new SqlCommand(sql, conn))
                {
                    cmd.ExecuteNonQuery();
                }
            }
        }
        /// <summary>
        /// Execute SQL command
        /// </summary>
        /// <param name="sql">text of command</param>
        internal static void ExecuteCommand(String sql)
        {
            ExecuteCommand(ConnectionString, sql);
        }

        /// <summary>
        /// Clear SQL table 
        /// </summary>
        /// <param name="connectionString">connection string to database</param>
        internal static void ClearTable(String connectionString)
        {
            String sql = @"TRUNCATE TABLE [dbo].[Contact]";

            ExecuteCommand(sql);
        }

        /// <summary>
        /// Insert in SQL table default values
        /// </summary>
        internal static void InsertSampleData()
        {
            Assembly executingAssembly = Assembly.GetExecutingAssembly();

            using (Stream stream = executingAssembly.GetManifestResourceStream("AddressValidation.DataLayer.DataSources.ContactList"))
            using (ZipArchive archive = new ZipArchive(stream, ZipArchiveMode.Read))
            {
                ZipArchiveEntry entry = archive.Entries.FirstOrDefault();

                if (entry == null)
                {
                    return;
                }

                Stream dataStream = entry.Open();
                XmlDocument xmlDoc = new XmlDocument();

                xmlDoc.Load(dataStream);

                String xPath = "NewDataSet/ContactList";

                var nodes = xmlDoc.SelectNodes(xPath);

                List<ContactInfo> contactInfos = new List<ContactInfo>(nodes.Count);

                foreach (XmlNode node in nodes)
                {
                    ContactInfo contact = new ContactInfo
                    {
                        FirstName = node["FirstName"]?.FirstChild?.Value,
                        CommonName = node["CommonName"]?.FirstChild?.Value,
                        LastName = node["LastName"]?.FirstChild?.Value,
                        Address1 = node["Address1"]?.FirstChild?.Value,
                        Address2 = node["Address2"]?.FirstChild?.Value,
                        City = node["City"]?.FirstChild?.Value,
                        State = node["State"]?.FirstChild?.Value,
                        Zip = node["Zip"]?.FirstChild?.Value,
                        Country = node["Country"]?.FirstChild?.Value,
                        Phone = node["Phone"]?.FirstChild?.Value,
                        Email = node["Email"]?.FirstChild?.Value,
                        Custom1 = String.Empty,
                        Custom2 = String.Empty,
                        IsDuplicate = false,
                        CreateTime = DateTime.Now,
                        IsVerified = "V".Equals(node["Status"]?.FirstChild?.Value, StringComparison.OrdinalIgnoreCase)
                    };

                    contactInfos.Add(contact);
                }

                AddContacts(contactInfos);
            }
        }

        /// <summary>
        /// Add one record into SQL table
        /// </summary>
        /// <param name="contactInfo">inserted record</param>
        public static void AddContact(ContactInfo contactInfo)
        {
            using (AddressDbModelDataContext dataContext = new AddressDbModelDataContext(Properties.Settings.Default.ContactDataDbConnectionString))
            {
                Contact contact = ContactInfo2Contact(contactInfo);
                dataContext.Contacts.InsertOnSubmit(contact);
                dataContext.SubmitChanges();
            }
        }

        /// <summary>
        /// Add a list of records in SQL table
        /// </summary>
        /// <param name="contactInfos">list of inserted records</param>
        public static void AddContacts(IEnumerable<ContactInfo> contactInfos)
        {
            using (AddressDbModelDataContext dataContext = new AddressDbModelDataContext(Properties.Settings.Default.ContactDataDbConnectionString))
            {
                foreach (var contactInfo in contactInfos)
                {
                    Contact contact = ContactInfo2Contact(contactInfo);
                    dataContext.Contacts.InsertOnSubmit(contact);
                }          
            
                dataContext.SubmitChanges();            
            }
        }

        /// <summary>
        /// Read all records from database
        /// </summary>
        /// <returns>list of records from SQL table</returns>
        public static List<ContactInfo> GetContacts()
        {
            return GetContacts(null, null, null);
        }

        /// <summary>
        /// Read records from database
        /// </summary>
        /// <param name="isDuplicate">if true - read all duplicates, false - all unique records</param>
        /// <returns>list of records from SQL table</returns>
        public static List<ContactInfo> GetContacts(Boolean isDuplicate)
        {
            return GetContacts(isDuplicate, null, null);
        }

        /// <summary>
        /// Read records from database
        /// </summary>
        /// <param name="startIndex">start index for reading</param>
        /// <param name="count">count for reading</param>
        /// <returns>list of records from SQL table</returns>
        public static List<ContactInfo> GetContacts(Int32 startIndex, Int32 count)
        {
            return GetContacts(null, startIndex, count);
        }

        /// <summary>
        /// Read records from database
        /// </summary>
        /// <param name="isDuplicate">if true - read only duplicates, false - unique records</param>
        /// <param name="startIndex">start index for reading</param>
        /// <param name="count">count for reading</param>
        /// <returns>list of records from SQL table</returns>
        public static List<ContactInfo> GetContacts(Int32 startIndex, Int32 count, Boolean isDuplicate)
        {
            return GetContacts(isDuplicate, startIndex, count);
        }

        /// <summary>
        /// Read records from database
        /// </summary>
        /// <param name="isDuplicate">if true - read only duplicates, false - unique records, null - both</param>
        /// <param name="startIndex">start index for reading</param>
        /// <param name="count">count for reading</param>
        /// <returns>list of records from SQL table</returns>
        private static List<ContactInfo> GetContacts(Boolean? isDuplicate, Int32? startIndex = null, Int32? count = null)
        {
            List<ContactInfo> contactInfos = new List<ContactInfo>();

            using (AddressDbModelDataContext dataContext = new AddressDbModelDataContext(Properties.Settings.Default.ContactDataDbConnectionString))
            {
                IQueryable<Contact> query;

                if (startIndex.HasValue && count.HasValue)
                {
                    query = isDuplicate.HasValue
                        ? dataContext.Contacts.Where(c => c.IsDuplicate == isDuplicate.Value).Skip(startIndex.Value).Take(count.Value)
                        : dataContext.Contacts.Skip(startIndex.Value).Take(count.Value);
                }
                else
                {
                    query = isDuplicate.HasValue
                        ? dataContext.Contacts.Where(c => c.IsDuplicate == isDuplicate.Value)
                        : dataContext.Contacts;
                }
                
                foreach (var contact in query)
                {
                    ContactInfo contactInfo = Contact2ContactInfo(contact);
                    contactInfos.Add(contactInfo);
                }
            }

            return contactInfos;
        }

        /// <summary>
        /// Convertion from application type to database type
        /// </summary>
        /// <param name="contactInfo"></param>
        /// <returns></returns>
        private static Contact ContactInfo2Contact(ContactInfo contactInfo)
        {
            Contact contact = new Contact
            {
                FirstName = contactInfo.FirstName,
                CommonName = contactInfo.CommonName,
                LastName = contactInfo.LastName,
                Address1 = contactInfo.Address1,
                Address2 = contactInfo.Address2,
                City = contactInfo.City,
                State = contactInfo.State,
                Zip = contactInfo.Zip,
                Country = contactInfo.Country,
                Phone = contactInfo.Phone,
                Email = contactInfo.Email,
                Custom1 = contactInfo.Custom1,
                Custom2 = contactInfo.Custom2,
                IsDuplicate = contactInfo.IsDuplicate,
                IsVerified = contactInfo.IsVerified,
                CreateTime = contactInfo.CreateTime
            };

            return contact;
        }

        /// <summary>
        /// Convertion from database type to application type
        /// </summary>
        /// <param name="contact">Entity from database</param>
        /// <returns>Model for application</returns>
        private static ContactInfo Contact2ContactInfo(Contact contact)
        {
            ContactInfo contactInfo = new ContactInfo
            {
                FirstName = contact.FirstName,
                CommonName = contact.CommonName,
                LastName = contact.LastName,
                Address1 = contact.Address1,
                Address2 = contact.Address2,
                City = contact.City,
                State = contact.State,
                Zip = contact.Zip,
                Country = contact.Country,
                Phone = contact.Phone,
                Email = contact.Email,
                Custom1 = contact.Custom1,
                Custom2 = contact.Custom2,
                IsDuplicate = contact.IsDuplicate,
                IsVerified = contact.IsVerified,
                CreateTime = contact.CreateTime
            };

            return contactInfo;
        }

        /// <summary>
        /// Check that defined in app.config or in web.config database exists
        /// </summary>
        /// <param name="connectionString">connection string</param>
        /// <param name="databaseName">database name</param>
        /// <returns></returns>
        private static Boolean DatabaseExists(String connectionString, String databaseName)
        {
            String sql = $"SELECT database_id FROM sys.databases WHERE Name = '{databaseName}'";
            
            using (SqlConnection conn = new SqlConnection(connectionString))
            using (SqlCommand cmd = new SqlCommand(sql, conn))
            {
                conn.Open();

                Object resultObj = cmd.ExecuteScalar();

                Int32 databaseId = 0;

                if (resultObj != null)
                {
                    Int32.TryParse(resultObj.ToString(), out databaseId);
                }

                conn.Close();

                return databaseId > 0;
            }
        }

        /// <summary>
        /// Create database that is described in app.config or web.config
        /// </summary>
        /// <param name="connectionString">connection string from configuration file</param>
        /// <param name="dbName">database name from configuration file</param>
        private static void CreateDatabase(String connectionString, String dbName)
        {
            ExecuteCommand(connectionString, $"CREATE DATABASE {dbName}");
        }
    }
}
