﻿using System.Collections.Generic;
using System.Data;

namespace AddressValidation.DataLayer.Entities
{
    /// <summary>
    /// This class contains a few helper methods for processing data from master SQL table
    /// </summary>
    public class ContactInfoHelper
    {
        /// <summary>
        /// Creates Contact Info from DataTable with the cleansed record.
        /// The cleansed record can have extra columns so this method 
        /// shrinks data, it filters the data only from needed columns.
        /// </summary>
        /// <param name="table">the cleansed data</param>
        /// <param name="isDuplicate">does master table contain similar data?</param>
        /// <param name="isVerified">were the cleansed data successfuly address verified?</param>
        /// <returns>Cleansed data in the type that can be easy inserted into master table</returns>
        public static ContactInfo CreateContactInfo(DataTable table, bool isDuplicate, bool isVerified)
        {
            var ci = new ContactInfo();

            ci.FirstName = Extract(table, "FirstName", "First Name");
            ci.CommonName = Extract(table, "Common Name");
            ci.LastName = Extract(table, "LastName", "Last Name");
            ci.Address1 = Extract(table, "V Primary Address", "Address1");
            ci.Address2 = Extract(table, "Address2", "V Secondary Address");
            ci.City = Extract(table, "V City", "City");
            ci.State = Extract(table, "V State", "State");
            ci.Zip = Extract(table, "V Zip Code", "Zip");
            ci.Country = Extract(table, "V Country", "Country");
            ci.Phone = Extract(table, "Phone");
            ci.Email = Extract(table, "Email");
            ci.Custom1 = Extract(table, "Custom1");
            ci.Custom2 = Extract(table, "Custom2");

            ci.CreateTime = System.DateTime.Now;

            ci.IsDuplicate = isDuplicate;
            ci.IsVerified = isVerified;

            return ci;
        }

        /// <summary>
        ///  Extract from the first row of the table a value from a column 
        ///  that has less index in params
        /// </summary>
        /// <param name="dt"></param>
        /// <param name="columnNames"></param>
        /// <returns></returns>
        private static string Extract(DataTable dt, params string[] columnNames)
        {
            string res = null;
            if (dt.Rows.Count > 0)
            {
                foreach (var name in columnNames)
                {
                    if (dt.Columns.Contains(name))
                    {
                        res = dt.Rows[0][name]?.ToString();
                        if (!string.IsNullOrEmpty(res))
                            break;
                    }
                }
            }
            return res;
        }

        /// <summary>
        /// Loads list of records from the master table to DataTable type
        /// that is comfortable for GUI working
        /// </summary>
        /// <param name="list">list of records from the master table</param>
        /// <param name="orderBack">change order of the list</param>
        /// <returns>type that is comfortable for GUI working</returns>
        public static DataTable ListContactInfoToDataTable(List<ContactInfo> list, System.Boolean orderBack)
        {
            if (orderBack)
                list.Reverse();

            var res = new DataTable("Contacts");
            res.Columns.Add("FirstName", typeof(string));
            res.Columns.Add("CommonName", typeof(string));
            res.Columns.Add("LastName", typeof(string));
            res.Columns.Add("Address1", typeof(string));
            res.Columns.Add("Address2", typeof(string));
            res.Columns.Add("City", typeof(string));
            res.Columns.Add("State", typeof(string));
            res.Columns.Add("Zip", typeof(string));
            res.Columns.Add("Country", typeof(string));
            res.Columns.Add("Phone", typeof(string));
            res.Columns.Add("Email", typeof(string));
            res.Columns.Add("Custom1", typeof(string));
            res.Columns.Add("Custom2", typeof(string));
            res.Columns.Add("CreateTime", typeof(System.DateTime));
            res.Columns.Add("VerifiedAddress", typeof(string));

            foreach (var contact in list)
            {
                var row = res.NewRow();
                row["FirstName"] = contact.FirstName;
                row["CommonName"] = contact.CommonName;
                row["LastName"] = contact.LastName;
                row["Address1"] = contact.Address1;
                row["Address2"] = contact.Address2;
                row["City"] = contact.City;
                row["State"] = contact.State;
                row["Zip"] = contact.Zip;
                row["Country"] = contact.Country;
                row["Phone"] = contact.Phone;
                row["Email"] = contact.Email;
                row["Custom1"] = contact.Custom1;
                row["Custom2"] = contact.Custom2;
                row["CreateTime"] = contact.CreateTime;
                row["VerifiedAddress"] = contact.IsVerified;
                res.Rows.Add(row);
            }

            return res;
        }

        /// <summary>
        /// Removes from DataTable the columns  that is absent in master SQL table
        /// </summary>
        /// <param name="table">DataTable with extra columns</param>
        /// <param name="verified">value for filling column 'IsVerified'</param>
        /// <returns>DataTable with same column set that in master table</returns>
        public static DataTable Normilize(DataTable table, bool verified =  false)
        {
            var ci = CreateContactInfo(table, false, false);
            ci.IsVerified = verified;
            return ListContactInfoToDataTable(new List<ContactInfo>() { ci }, false);
        }
    }
}
