﻿using dataladder.Data;
using dataladder.Matching;
using DataMatch.Project.Descriptors.MatchDefinition;
using System;
using System.Collections.Generic;

namespace Samples.Common.Builders
{
    /// <summary>
    /// This class creates a instance of 'MultipleMatchDefinitionsManager'
    /// (that is needed for creating Match Engine)
    /// from 'MatchDefinitionSpec' (that can be founded in project file). 
    /// 
    /// This class links definitions and list of sources, 
    /// also defines which fields of these sources
    /// will be matched
    /// </summary>
    public class DefinitionManagerBuilder 
    {
        private readonly MatchDefinitionSpec _definitionSpec;
        private readonly List<ITable2CoordsMapper> _tables;

        /// <summary>
        /// Create instance of class.
        /// </summary>
        /// <param name="definitionSpec">specifications of definitions that was extracted from a project file</param>
        /// <param name="tables">list of data sources for which matching rules are defined </param>
        public DefinitionManagerBuilder(MatchDefinitionSpec definitionSpec, List<ITable2CoordsMapper> tables)
        {
            _definitionSpec = definitionSpec;
            _tables = tables;
        }

        /// <summary>
        /// Creates a instance of 'MultipleMatchDefinitionsManager'
        /// (that is needed for creating Match Engine)
        /// by parameters that were transferred  to this class in the constructor
        /// </summary>
        /// <returns></returns>
        public MultipleMatchDefinitionsManager Build()
        {
            MultipleMatchDefinitionsManager resultDefinitionManager = new MultipleMatchDefinitionsManager();
            Build(resultDefinitionManager);
            return resultDefinitionManager;
        }

        /// <summary>
        /// Fills specified 'MultipleMatchDefinitionsManager' with matching settings
        /// that were transferred to this class in the constructor
        /// </summary>
        /// <param name="definitionManager">'MultipleMatchDefinitionsManager' that will be changed </param>
        public void Build(MultipleMatchDefinitionsManager definitionManager)
        {
            // 1. fills 'AvailableFields' - how columns of tables will map each other
            definitionManager.AvailableFields = new AvailableFields();

            var fieldInfoMapping = new Dictionary<String, Dictionary<String, FieldMapInfo>>();

            for (Int32 tableIndex = 0; tableIndex < _tables.Count; tableIndex++)
            {
                ITable2CoordsMapper table = _tables[tableIndex];
                AvailableFieldsFromOneTable affot = new AvailableFieldsFromOneTable {Table = table};
                var field2FieldMapInfo = new Dictionary<String, FieldMapInfo>();
                fieldInfoMapping.Add(table.Name, field2FieldMapInfo);

                for (Int32 colIndex = 0; colIndex < table.ColumnCount; colIndex++)
                {
                    String fieldName = table.GetColumnName(colIndex);
                    FieldMapInfo fmi = new FieldMapInfo(tableIndex)
                    {
                        FieldName = fieldName,
                        TableName = table.Name,
                        ColumnTransformation = null,
                        FieldIndex = colIndex
                    };
                    affot.Add(fmi);
                    field2FieldMapInfo.Add(fieldName.ToUpper(), fmi);
                }

                definitionManager.AvailableFields.TableList.Add(affot);
            }

            for (Int32 rowIndex = 0; rowIndex < _definitionSpec.MappedFields.MappedFields.Count; rowIndex++)
            {
                MappedFieldsRow mfr = new MappedFieldsRow();
                MappedFieldSpec fieldSpec = _definitionSpec.MappedFields.MappedFields[rowIndex];
                
                for (Int32 tableIndex = 0; tableIndex < _tables.Count; tableIndex++)
                {
                    String tableName = definitionManager.AvailableFields.GetTableName(tableIndex);
                    String fieldName;
                    FieldMapInfo potentialFieldMapInfo = null;

                    if (fieldSpec.MappedFields.TryGetValue(tableName, out fieldName))
                    {
                        if (fieldInfoMapping[tableName].TryGetValue(fieldName.ToUpper(), out potentialFieldMapInfo))
                        {
                            mfr[tableName] = potentialFieldMapInfo;
                        }
                    }

                    if (potentialFieldMapInfo == null)
                    {
                        mfr[tableName] = new FieldMapInfo(tableIndex);
                    }
                }
                definitionManager.AvailableFields.MappedFieldsRowList.Add(mfr);
            }

            if (_tables.Count > 1)
            {
                MappedFieldsRow lastMfr = new MappedFieldsRow();

                for (Int32 tableIndex = 0; tableIndex < _tables.Count; tableIndex++)
                {
                    String tableName = definitionManager.AvailableFields.GetTableName(tableIndex);
                    lastMfr[tableName] = new FieldMapInfo(tableIndex);
                }
            }

            definitionManager.SetAbsoluteIndices();

            // 2. fills match definitions
            for (Int32 mdIndex = 0; mdIndex < _definitionSpec.Definitions.Definitions.Count; mdIndex++)
            {
                MatchCriteriaList matchDefinition = new MatchCriteriaList(mdIndex);

                List<MatchCriteriaSpec> criteriaSpecs = _definitionSpec.Definitions.Definitions[mdIndex].MatchCriteriaList;

                for (Int32 criteriaIndex = 0; criteriaIndex < criteriaSpecs.Count; criteriaIndex++)
                {
                    MatchCriteriaSpec mcs = criteriaSpecs[criteriaIndex];
                    MatchCriteria matchCriteria = new MatchCriteria()
                    {
                        Fuzzy = mcs.Fuzzy,
                        AddWeightToFirstLetter = mcs.AddWeightToFirstLetter,
                        Exact = mcs.Exact,
                        Numeric = mcs.Numeric,
                        UseMetaphone = mcs.UseMetaphone,
                        IgnoreCase = mcs.IgnoreCase,
                        Level = mcs.Level,
                        GroupLevel = mcs.GroupLevel,
                        MinAllowedLevelInGroup = mcs.MinAllowedLevelInGroup,
                        GroupId = mcs.GroupId,
                        CrossColumnGroupId = mcs.CrossColumnGroupId,
                        MaxTotalWeightBelow = mcs.MaxTotalWeightBelow,
                        MaxMismatchWeightBelow = mcs.MaxMismatchWeightBelow,
                        MaxEmptyWeightBelow = mcs.MaxEmptyWeightBelow,
                        Weight = mcs.Weight,
                        MatchingIndex = mcs.MatchingIndex,
                        AbsoluteMatchingIndex = mcs.AbsoluteMatchingIndex
                    };

                    foreach (var mapping in mcs.TableFieldDictionary)
                    {
                        matchCriteria.MapField(mapping.Key, mapping.Value);
                    }
                    matchDefinition.Add(matchCriteria);
                }

                matchDefinition.MarkTheFirstFieldInEveryGroup();
                definitionManager.Add(matchDefinition);
            }

            definitionManager.SetAbsoluteIndices();
        }

    }
}
