﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using dataladder.Data.DataTransformation;
using DataMatch.Project.Descriptors.DataSource;

namespace Samples.Common.Builders
{
    /// <summary>
    /// Extracts from project file standardization settings.
    /// 
    /// </summary>
    public class TransformationBuilder
    {
        private ColumnTransformationsSpec _standardizedInfo;
        private MergeInfoSpec _mergingInfo;
        private String _tempPath;

        /// <summary>
        /// Create instance
        /// </summary>
        /// <param name="standardizedSpec">Specification of column transformations</param>
        /// <param name="mergingSpec">Specification of merging rules</param>
        /// <param name="tempPath">Path to the temporary folder</param>
        public TransformationBuilder(ColumnTransformationsSpec standardizedSpec, MergeInfoSpec mergingSpec, string tempPath)
        {
            _standardizedInfo = standardizedSpec;
            _mergingInfo = mergingSpec;
            _tempPath = tempPath;
        }

        /// <summary>
        /// Create column transformation rules for all columns
        /// </summary>
        /// <returns></returns>
        public ColumnTransformationList BuildStandardizedSettings()
        {
            ColumnTransformationList res = new ColumnTransformationList(_tempPath);

            List<ColumnTransformationSpec> list = _standardizedInfo.Transformations;
            foreach (var specification in list)
            {
                ColumnTransformation transform = ConvertToTransformation(specification);
                res.Add(transform);
            }

            return res;
        }

        /// <summary>
        /// Create merging rules 
        /// </summary>
        /// <returns></returns>
        public MergeBlocksStorage BuildMergingSettings()
        {
            MergeBlocksStorage res = new MergeBlocksStorage();
            foreach (var item in _mergingInfo.MergedFields)
            {
                string name = item.FieldName;
                List<string> mergedFields = item.MergedFields;
                MergeBlockSingleStorage single = new MergeBlockSingleStorage(name, mergedFields);

                //remove 'garbage char'
                single.Delimiter = _mergingInfo.Delimiter.Replace("Z", "");

                res.Add(single);
            }
            return res;
        }

        /// <summary>
        /// Create column  transformation rules only for one column
        /// </summary>
        /// <param name="spec"></param>
        /// <returns></returns>
        private ColumnTransformation ConvertToTransformation(ColumnTransformationSpec spec)
        {
            ColumnTransformation res = new ColumnTransformation(_tempPath);

            res.FieldName = spec.FieldName;

            res.TransformationType = TransformationTypes.None;
            if (Enum.TryParse<TransformationTypes>(spec.Type, out TransformationTypes typez))
            {
                res.TransformationType = typez;
            }

            res.CopyField = spec.CopyField;
            res.ChangeCase = spec.ChangeCase;
            res.UpperCase = spec.UpperCase;
            res.LowerCase = spec.LowerCase;

            res.ProperCaseSettings = (ProperCaseSettings)spec.ProperCase;
            res.ProperCase = res.ProperCaseSettings.UseProperCase;

            res.RemoveNonPrintableCharacters = spec.RemoveNonPrintableCharacters;
            res.ReplacementForNonPrintableCharacters = spec.ReplacementForNonPrintableCharacters;
            res.ReplacementForEmptyValues = spec.ReplacementForEmptyValues;
            res.RemoveLeadingSpaces = spec.RemoveLeadingSpaces;
            res.RemoveTrailingSpaces = spec.RemoveTrailingSpaces;
            res.CharactersToRemove = spec.CharactersToRemove;
            if (string.IsNullOrEmpty(spec.CharactersToReplaceCaseSensitive))
                res.CharactersToReplace = spec.CharactersToReplace;
            else
                res.CharactersToReplace = spec.CharactersToReplaceCaseSensitive;
            res.RemoveSpaces = spec.RemoveSpaces;
            res.RemoveLetters = spec.RemoveLetters;
            res.RemoveDigits = spec.RemoveDigits;
            res.ReplaceZerosWithOs = spec.ReplaceZerosWithOs;
            res.ReplaceOsWithZeros = spec.ReplaceOsWithZeros;
            res.RegExSettings.Expression = spec.Regex;
            String path = (spec.WordSmith != null) ? spec.WordSmith : "";
            if (!String.IsNullOrEmpty(path))
            {
                res.WordSmithVisualizator.StorageText = path;
            }

            return res;
        }        
    }
}
