﻿using System;
using dataladder.AddressVerification;
using dataladder.Data;
using dataladder.Data.DataTransformation;
using DataMatch.AddressVerification.Cass;
using DataMatch.Transformation.Blocks.RegexBlock;

namespace Samples.Common.Builders
{
    /// <summary>
    /// Class that creates satandardization scheme - transformation diagram
    /// </summary>
    public class TransformationDiagramBuilder
    {
        private readonly ColumnTransformationList _columnTransformationList;
        private readonly MergeBlocksStorage _mergeBlocksStorage;
        private readonly String _tempPath;
        private readonly String _cassPath, _geoPath;
        private readonly OnDriveTable _mapper;
        private readonly AddressVerificationSettings _addressVerificationSettings;

        private static dataladder.Parsers.AddressParser s_addressParsers;
        private static dataladder.Parsers.FullNameParser s_FullNameParsers;
        private static dataladder.Parsers.AbbreviationParser s_abbreviationParsers;
        private static DataMatch.Transformation.ProperCase.ProperCaseOptions s_properCaseOptions;
        private static CassAddress s_cassAddress = new CassAddress();
        private static CassGeoCoder s_cassGeoCoder = new CassGeoCoder();

        /// <summary>
        /// Create instance
        /// </summary>
        /// <param name="tempPath">path to temporary folder</param>
        /// <param name="cassPath">path to folder with CASS database</param>
        /// <param name="geoPath">path to folder with CASS Geo coder database</param>
        /// <param name="transformations">transformation settings for all column</param>
        /// <param name="mergeStorage">merging settings</param>
        /// <param name="mapper">data storage on which will be applied standardization</param>
        /// <param name="avSettings">CASS settings which columns of adress will appear after address verification</param>
        public TransformationDiagramBuilder(String tempPath, String cassPath, String geoPath,
            ColumnTransformationList transformations, MergeBlocksStorage mergeStorage, OnDriveTable mapper,
            AddressVerificationSettings avSettings = null)
        {
            _columnTransformationList = transformations;
            _mergeBlocksStorage = mergeStorage;
            _tempPath = tempPath;
            _cassPath = cassPath;
            _geoPath = geoPath;
            _mapper = mapper;
            _addressVerificationSettings = avSettings?? new AddressVerificationSettings();
        }

        /// <summary>
        /// Create transformation diagram
        /// </summary>
        /// <returns></returns>
        public TransformationDiagram Build()
        {
            TransformationDiagram diagram = new TransformationDiagram(0);

            AddressBlock addressBlock = null;
            AddressCassBlock cassBlock = null;
            Int32 regexOrdinal = 0;

            for (Int32 i = 0; i < _columnTransformationList.Count; i++)
            {
                ColumnTransformation columnTransformation = _columnTransformationList[i];
                DataFlow input = AddInput(columnTransformation.FieldName, diagram);

                if (input == null) continue;

                if (columnTransformation.CopyField)
                {
                    CopyColumnBlock copyColumnBlock = new CopyColumnBlock(input);
                    diagram.AddTransformationBlock(copyColumnBlock);
                }

                if (columnTransformation.Active)
                {
                    InitAbbreviationParsersIfNeeded();
                    InitProperCaseOptionsIfNeeded();
                    input = addCleanBlock(diagram, input, columnTransformation, s_abbreviationParsers, s_properCaseOptions);
                }

                if (columnTransformation.UseWordSmith)
                {
                    WordSmithBlock wordSmithBlock = new WordSmithBlock(
                        columnTransformation.WordSmithVisualizator.WordsToReplaceDictionary,
                        columnTransformation.WordSmithVisualizator.NewColumnsDictionary,
                        columnTransformation.WordSmithVisualizator.MaxCount,
                        columnTransformation.WordSmithVisualizator.SeparatorChars);
                    diagram.AddTransformationBlock(wordSmithBlock);
                    wordSmithBlock.SetInput(0, input);
                    input = wordSmithBlock.Outputs[0];
                }

                if (dataladder.Licensing.Registration.UseAdressVerification)
                {
                    if ((Int32)columnTransformation.TransformationType >= (Int32)TransformationTypes.VCompanyName)
                    {
                        if (cassBlock == null)
                        {
                            InitAccuCassIfNeeded(_cassPath, _geoPath);

                            cassBlock = new AddressCassBlock(s_cassAddress, s_cassGeoCoder, _addressVerificationSettings);

                            diagram.AddTransformationBlock(cassBlock);
                            diagram.ContainsCassBlock = true;
                        }
                    }
                }

                switch (columnTransformation.TransformationType)
                {
                    case TransformationTypes.Address:
                        if (addressBlock == null)
                        {
                            InitAddressParsersIfNeeded();
                            addressBlock = new AddressBlock(s_addressParsers);
                            diagram.AddTransformationBlock(addressBlock);
                        }
                        addressBlock.AddInput(input);
                        break;
                    case TransformationTypes.FullName:
                        InitFullNameParsersIfNeeded();
                        FullNameBlock fullNameBlock = new FullNameBlock(s_FullNameParsers);
                        diagram.AddTransformationBlock(fullNameBlock);
                        fullNameBlock.AddInput(input);
                        break;
                    case TransformationTypes.FirstName:
                        FirstNameBlock firstNameBlock = AddFirstNameBlock(diagram);
                        firstNameBlock.AddInput(input);
                        break;
                    case TransformationTypes.Zip:
                        ZipBlock zipBlock = new ZipBlock();
                        diagram.AddTransformationBlock(zipBlock);
                        zipBlock.AddInput(input);
                        break;
                    default:
                        if (cassBlock != null)
                        {
                            switch (columnTransformation.TransformationType)
                            {
                                case TransformationTypes.VCityName:
                                    cassBlock.AddInput(input, CassInputTypes.CityName);
                                    break;
                                case TransformationTypes.VCompanyName:
                                    cassBlock.AddInput(input, CassInputTypes.CompanyName);
                                    break;
                                case TransformationTypes.VPrimaryAddress:
                                    cassBlock.AddInput(input, CassInputTypes.PrimaryAddress);
                                    break;
                                case TransformationTypes.VSecondaryAddress:
                                    cassBlock.AddInput(input, CassInputTypes.SecondaryAddress);
                                    break;
                                case TransformationTypes.VStateName:
                                    cassBlock.AddInput(input, CassInputTypes.StateName);
                                    break;
                                case TransformationTypes.VUrbanization:
                                    cassBlock.AddInput(input, CassInputTypes.Urbanization);
                                    break;
                                case TransformationTypes.VZipCode:
                                    cassBlock.AddInput(input, CassInputTypes.ZipCode);
                                    break;
                                case TransformationTypes.VCountry:
                                    cassBlock.AddInput(input, CassInputTypes.Country);
                                    break;
                                case TransformationTypes.VPostalCode:
                                    cassBlock.AddInput(input, CassInputTypes.PostalCode);
                                    break;
                            }
                        }

                        break;
                }
                //#endif

                if (!String.IsNullOrEmpty(columnTransformation.RegExSettings.Expression))
                {
                    columnTransformation.RegExSettings.FormattedFieldName = input.Name + "_formatted";
                    RegexBlock regexBlock = RegexBlockFactory.Create(columnTransformation.RegExSettings, null, regexOrdinal++);
                    regexBlock.Inputs[0] = input;
                    diagram.AddTransformationBlock(regexBlock);
                }
            }

            diagram.CreateOutputs();

            if (_mergeBlocksStorage != null)
            for (Int32 i = 0; i < _mergeBlocksStorage.Count; i++)
            {
                MergeBlockSingleStorage mergeBlockSingleStorage = _mergeBlocksStorage[i];

                if (mergeBlockSingleStorage.MergedFieldNames.Count > 0)
                {
                    MergeBlock mergeBlock = new MergeBlock(mergeBlockSingleStorage.Name);
                    diagram.AddTransformationBlock(mergeBlock);
                    mergeBlock.Delimiter = mergeBlockSingleStorage.Delimiter;
                    mergeBlock.OnlyFirstNotEmptyCount = mergeBlockSingleStorage.OnlyFirstNotEmptyCount;

                    foreach (String name in mergeBlockSingleStorage.MergedFieldNames)
                    {
                        DataFlow input = diagram.Outputs[name];

                        if (input != null)
                        {
                            mergeBlock.AddInput(input);
                        }
                    }
                }
            }

            diagram.CreateOutputs();

            return diagram;
        }

        /// <summary>
        /// Add to diagramm input for specified column
        /// </summary>
        /// <param name="fieldName">name of column</param>
        /// <param name="diagram">diagram to which will be added new input</param>
        /// <returns></returns>
        private DataFlow AddInput(String fieldName, TransformationDiagram diagram)
        {
            OnDriveField field = _mapper[fieldName];
            DataFlow result = null;
            if (field != null)
            {
                result = diagram.AddInput(fieldName, field.Index);
            }

            return result;
        }

        /// <summary>
        /// Init Address Parser
        /// </summary>
        private void InitAddressParsersIfNeeded()
        {
            if (s_addressParsers == null)
            {
                    s_addressParsers = new dataladder.Parsers.AddressParser();
            }
        }
        
        /// <summary>
        /// Init dictionaries with abbreviations
        /// </summary>
        private void InitAbbreviationParsersIfNeeded()
        {
            if (s_abbreviationParsers == null)
            {
                s_abbreviationParsers = new dataladder.Parsers.AbbreviationParser();
            }
        }

        /// <summary>
        /// Init full name parser
        /// </summary>
        private void InitFullNameParsersIfNeeded()
        {
            if (s_FullNameParsers == null)
            {               
                s_FullNameParsers = new dataladder.Parsers.FullNameParser();                
            }
        }
        /// <summary>
        /// Init Proper Case Parser
        /// </summary>
        private void InitProperCaseOptionsIfNeeded()
        {
            if (s_properCaseOptions == null)
            {
                s_properCaseOptions = new DataMatch.Transformation.ProperCase.ProperCaseOptions();
                s_properCaseOptions.Load();
            }
        }

        /// <summary>
        /// Create and load First Name Block
        /// </summary>
        /// <param name="diagram"></param>
        /// <returns></returns>
        private FirstNameBlock AddFirstNameBlock(TransformationDiagram diagram)
        {
            FirstNameBlock firstNameBlock = new FirstNameBlock();

            {
                dataladder.Data.Format.CommonNames.LoadDictionary();
                dataladder.Data.Format.Gender.LoadDictionary();
            }

            diagram.AddTransformationBlock(firstNameBlock);
            return firstNameBlock;
        }


        private DataFlow addCleanBlock(TransformationDiagram diagram, DataFlow input, 
            ColumnTransformation columnTransformation,
            dataladder.Parsers.AbbreviationParser abbreviationParser, 
            DataMatch.Transformation.ProperCase.ProperCaseOptions properCaseOptions)
        {
            CleanBlock cleanBlock = new CleanBlock(columnTransformation, abbreviationParser, properCaseOptions);
            cleanBlock.SetInput(0, input);
            diagram.AddTransformationBlock(cleanBlock);
            input = cleanBlock.Outputs[0];
            return input;
        }
               
        private static void InitAccuCassIfNeeded(String cassPath, String geoPath)
        {
            if (s_cassAddress == null)
            {
                s_cassAddress = new CassAddress();
            }

            Int32 returnCode = s_cassAddress.Init(cassPath);

            if (returnCode != 0)
            {
                String errorDescription = String.Empty;

                switch (returnCode)
                {
                    case 1:
                        errorDescription = DataMatch.AddressVerification.Properties.Resources.ErrorDescription1;
                        break;
                    case 2:
                        errorDescription = DataMatch.AddressVerification.Properties.Resources.ErrorDescription2;
                        break;
                    case 3:
                        errorDescription = DataMatch.AddressVerification.Properties.Resources.ErrorDescription3;
                        break;
                    case 4:
                        errorDescription = DataMatch.AddressVerification.Properties.Resources.ErrorDescription4;
                        break;
                    case 5:
                        errorDescription = DataMatch.AddressVerification.Properties.Resources.ErrorDescription5;
                        break;
                    case 6:
                        errorDescription = DataMatch.AddressVerification.Properties.Resources.ErrorDescription6;
                        break;
                    case 7:
                        errorDescription = DataMatch.AddressVerification.Properties.Resources.ErrorDescription7;
                        break;
                    default:
                        break;
                }
                
                throw new Exception(errorDescription);
            }

            if (s_cassGeoCoder == null)
            {
                s_cassGeoCoder = new CassGeoCoder();
            }
            s_cassGeoCoder.Init(geoPath);
        }

    }
}
