﻿using dataladder.XtraGridHelper;
using Samples.Common.Matching;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Samples.Common
{
    /// <summary>
    /// Set of simple functions that help to work with the console
    /// </summary>
    public static class ConsoleHelper
    {
        /// <summary>
        /// Shows in the console short information about matching settings
        /// </summary>
        /// <param name="matchSettings">matching settings</param>
        public static void PrintMatchSettings(MatchSettings matchSettings)
        {
            Console.WriteLine("Initializing matching settings...");

            Console.WriteLine("Data Path: \t\t {0}", matchSettings.DataPath);
            Console.WriteLine("Temp Path: \t\t {0}", matchSettings.TempPath);
            Console.WriteLine("Project Name: \t\t {0}", matchSettings.Name);

            Console.WriteLine();
        }

        /// <summary>
        /// Prints in console content of data source.
        /// </summary>
        /// <param name="mapper">data source that will be printed</param>
        /// <param name="width">width of columns in symbols</param>
        /// <param name="toRow">how many rows will be printed</param>
        /// <param name="widths">individual column lengths</param>
        public static void PrintTable(IContainer2CoordsMapper mapper, Int32 width, Int32 toRow = Int32.MaxValue, Int32[] widths = null)
        {
            Int32 lineLength = mapper.ColumnCount * (width + 1) - 1;
            if (widths != null && widths.Length > 0)
            {
                lineLength = widths.Select(w => w + 1).Sum() - 1;
            }

            String line = "+" + new String('-', lineLength) + "+";
            Console.WriteLine(line);

            for (Int32 i = 0; i < mapper.ColumnCount; i++)
            {
                if (widths != null && widths.Length > 0)
                {
                    width = widths[i];
                }

                String columnName = mapper.GetColumnName(i);
                if (columnName.Length > width - 2)
                {
                    columnName = columnName.Substring(0, width - 2) + (Char)(0x2026);
                }
                Console.Write("|");
                Console.Write($"{{0,-{width}}}", columnName);
            }

            Console.WriteLine("|");
            Console.WriteLine(line);

            for (Int32 j = 0; j < mapper.RecordCount && j <= toRow; j++)
            {
                for (Int32 i = 0; i < mapper.ColumnCount; i++)
                {
                    Object cellValue = mapper.GetData(j, i);
                    Double? doubleValue = cellValue as Double?;

                    String cellValueString;

                    if (doubleValue.HasValue)
                    {
                        cellValueString = doubleValue.Value.ToString("n2");
                    }
                    else
                    {
                        cellValueString = cellValue?.ToString() ?? String.Empty;
                    }

                    if (widths != null && widths.Length > 0)
                    {
                        width = widths[i];
                    }

                    if (cellValueString.Length > width - 2)
                    {
                        cellValueString = cellValueString.Substring(0, width - 2);
                    }
                    Console.Write("|");
                    Console.Write($"{{0,-{width}}}", cellValueString);
                }

                Console.WriteLine("|");
            }

            Console.WriteLine(line);
            Console.WriteLine();
        }
        /// <summary>
        /// Defines best width for each column in specified data source
        /// </summary>
        /// <param name="mapper">data source</param>
        /// <param name="maxWidth">max width of column</param>
        /// <param name="toRow">count of rows that will be taken for defining column widths, if it is not chosen then all rows will be scanned</param>
        /// <returns></returns>
        public static Int32[] GetBestColumnWidths(IContainer2CoordsMapper mapper, Int32 maxWidth, Int32 toRow = Int32.MaxValue)
        {
            Int32 colCount = mapper?.ColumnCount ?? 0;
            Int32 rowCount = mapper?.RecordCount ?? 0;

            Int32[] result = new Int32[colCount];

            for (Int32 i = 0; i < colCount; i++)
            {
                result[i] = 2;
            }

            for (Int32 i = 0; i < colCount; i++)
            {
                String columnName = mapper?.GetColumnName(i) ?? String.Empty;
                Int32 colWidth = columnName.Length + 2;

                if (colWidth > maxWidth)
                {
                    colWidth = maxWidth;
                }

                if (colWidth > result[i])
                {
                    result[i] = colWidth;
                }
            }

            for (Int32 j = 0; j < rowCount && j <= toRow; j++)
            {
                for (Int32 i = 0; i < colCount; i++)
                {
                    Object cellValue = mapper.GetData(j, i);
                    Double? doubleValue = cellValue as Double?;

                    String cellValueString;

                    if (doubleValue.HasValue)
                    {
                        cellValueString = doubleValue.Value.ToString("n2");
                    }
                    else
                    {
                        cellValueString = cellValue?.ToString() ?? String.Empty;
                    }

                    Int32 colWidth = cellValueString.Length + 2;

                    if (colWidth > maxWidth)
                    {
                        colWidth = maxWidth;
                    }

                    if (colWidth > result[i])
                    {
                        result[i] = colWidth;
                    }
                }
            }

            return result;
        }
    }
}
