﻿using dataladder.AddressVerification;
using dataladder.Data;
using dataladder.Data.DataTransformation;
using dataladder.IO;
using dataladder.Licensing;
using dataladder.Matching;
using DataMatch.AddressVerification.Cass;
using Samples.Common;
using Samples.Common.Matching;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading;

namespace Samples.MatchTransform
{
    /// <summary>
    /// Example with transformation and matching
    /// </summary>
    class Program
    {
        // path to folder that contains 'license.txt' - registration file
        private const String CustomPathForRegistrationFile = @"D:\DataMatchSDK\License";
        // path to folder that will contain working subfolder
        private const String DemoDataPath = @"D:\DataMatchSDK\Test";
        // temporary folder
        private const String DemoTempPath = @"D:\DataMatchSDK\Test\Temp";

        // Several samples can use this object.
        private static RegistrationWrapper _registration = new RegistrationWrapper();

        // Required for address transformation.
        private static CassParser _cassParser = new CassParser();

        static void Main(string[] args)
        {
            TestWith2Tables();

            Console.WriteLine("Press any key to continue . . .");
            Console.ReadKey();
        }

        public static void TestWith2Tables()
        {
            // Check registration
            if (!InitRegistration(CustomPathForRegistrationFile))
            {
                return;
            }

            // working subfolder
            String dataPath = @"D:\DataMatchSDK\Test\Test2Tables";
            // folder for keeping intermediate results
            String uncompletePath = Path.Combine(dataPath, "uncomplete");
            //Excel file that will be input source for this example
            String sourceFileName = @"D:\DataMatchSDK\Sources\ContactListMaster.xls";
            //Storage name for loading records from data source
            String sourceTableName = "ContactList";
            //Storage name for second data source
            String bufferTableName = "BufferTable";
            //Path to CASS database
            String addressDbPath = @"C:\enterprise";
            //Path to CASS Geocoder database
            String geoDbPath = @"C:\enterprise\Geocoder\Accugeo";
            //Storage name for keeping cleansed second source
            String transformedTableName = "BufferTransformed";

            IOHelper.CreateDirectoryIfNotExist(dataPath);
            IOHelper.CreateDirectoryIfNotExist(uncompletePath);
            IOHelper.CreateDirectoryIfNotExist(DemoTempPath);

            //Prepares for reading first data source from Excel file
            ReaderConfiguration readerConfiguration = new ReaderConfiguration();
            readerConfiguration.DataSourceTypeDescription = SourceDb.Excel.ToString();
            readerConfiguration.DataSourceId = 0;
            readerConfiguration.FileName = sourceFileName;
            readerConfiguration.TableName = sourceTableName;
            readerConfiguration.OriginalTableName = sourceTableName;

            // A little bit tricky because for the new ReaderConfiguration ColumnSettings = null;
            Fields fields = new Fields();
            readerConfiguration.ColumnsSettings = fields.SaveColumnsSettings();

            ////////////////////////////////////////////////////////////////
            // Reads data source
            // Loads records into intermediate storage
            // Copy lock into buffer
            ///////////////////////////////////////////////////////////////
            
            // intermediate storage:
            OnDriveTable inputTable = null;

            String error = String.Empty;
            Boolean errorOccured = false;
            try
            {
                IReader reader = dataladder.Data.IReaderHelper.CreateReader(readerConfiguration);
                reader.SetConfiguration(readerConfiguration);

                reader.ReadTable(readerConfiguration, toDetermineFields: true);

                //defines which columns will be present in output
                String[] fieldsToInclude =
                {
                    "Id",
                    "FirstName",
                    "LastName",
                    "Address1",
                    "Address2",
                    "City",
                    "State",
                    "Zip",
                    "Country",
                    "Phone",
                    "Email",
                    //"Custom1",
                    //"Custom2",
                    "IsDuplicate",
                    "IsVerified"
                };
                for (Int32 fieldIndex = 0; fieldIndex < reader.Fields.RecordCount; fieldIndex++)
                {
                    Field field = reader.Fields[fieldIndex];
                    field.Included = fieldsToInclude.FirstOrDefault(fn => fn.Equals(field.OriginalName, StringComparison.OrdinalIgnoreCase)) != null;
                }

                CancellationTokenSource tokenSource = new CancellationTokenSource();
                CancellationToken cancellationToken = tokenSource.Token;

                inputTable = ReaderToVariableTableConvertor.Copy(reader,
                    uncompletePath,
                    readerConfiguration.OriginalTableName, // could be changed if we have 2 or more same tables in the
                                                           // project, like Sheet1 and Sheet1_ (in other case TableName, doesn't work, but OriginalTableName do)
                    out error,
                    cancellationToken,
                    onTableCopyProgress: null,
                    fileNameBase: sourceTableName,
                    onMinSpaceReached: null,
                    operationMode: OperationModes.Disk);

                inputTable?.Move(dataPath);

                reader.Disconnect();

            }
            catch (Exception ex)
            {
                errorOccured = true;
                Console.WriteLine("Unable to read table.");
                Console.WriteLine(ex.Message);
            }

            if (errorOccured)
            {
                return;
            }

            Int32[] widths = ConsoleHelper.GetBestColumnWidths(inputTable, 30, 20);
            ConsoleHelper.PrintTable(inputTable, 15, 20, widths);

            // create second input storage  with same structure
            //and copy in it first row 

            OnDriveTable bufferTable = OnDriveTable.CreateNewTableWithTheStructureOfExisting(inputTable, bufferTableName, dataPath);
            for (Int32 i = 0; i < inputTable.ColumnCount; i++)
            {
                Object value = inputTable.GetData(0, i);
                bufferTable.SetData(value, 0, i);
            }

            ConsoleHelper.PrintTable(bufferTable, 15);

            ////////////////////////////////////////////////////
            // Transformation Part 
            ////////////////////////////////////////////////////

            Int32 currentRowIndex = 0;

            TransformationDiagram transformationDiagram = new TransformationDiagram(0);

            List<Int32> fieldIndexMapping = new List<Int32>
            {
                0,
                1, // FirstName
                2,
                3, // Address1
                4,
                5, // City
                6,
                7,  // Zip
                8,
                9,
                10,
                11,
                12
                //13,
                //14
            };

            transformationDiagram.OnGetInputData += (index, taskIndex, rowIdx) => bufferTable.GetData(rowIdx, fieldIndexMapping[index]);
            transformationDiagram.Clear();

            // 1. Transfer Id column
            transformationDiagram.AddInput("Id", 0);

            // 2. Initialize First name transformation
            dataladder.Data.Format.CommonNames.LoadDictionary();
            dataladder.Data.Format.Gender.LoadDictionary();

            FirstNameBlock firstNameBlock = new FirstNameBlock();
            transformationDiagram.AddTransformationBlock(firstNameBlock);
            DataFlow firstNameInput = transformationDiagram.AddInput("FirstName", 1);
            firstNameBlock.AddInput(firstNameInput);

            // 3. Transfer LastName column
            transformationDiagram.AddInput("LastName", 2);

            // 4. Initialize CASS transformation

            Console.WriteLine("Initializing CASS...");

            _cassParser.InitCassIfNeeded(addressDbPath, geoDbPath);

            // defines a set of column that appear after address verification
            AddressVerificationSettings addressVerificationSettings = new AddressVerificationSettings();
            addressVerificationSettings.OutputColumnSettings.ForEach(s =>
            {
                switch (s.CassOutputPart)
                {
                    case CassOutputParts.Status:
                    case CassOutputParts.CompanyFirm:
                    case CassOutputParts.PrimaryAddress:
                    case CassOutputParts.SecondaryAddress:
                    case CassOutputParts.City:
                    case CassOutputParts.State:
                    case CassOutputParts.ZipCode:
                    case CassOutputParts.PostalCode:
                    case CassOutputParts.BuildingNumber:
                    case CassOutputParts.StreetName:
                        s.Include = true;
                        break;
                    default:
                        s.Include = false;
                        break;
                }
            });

            // create block that will make address verification
            AddressCassBlock cassBlock = new AddressCassBlock(_cassParser.CassAddress, 
                _cassParser.CassGeoCoder, addressVerificationSettings);
            transformationDiagram.AddTransformationBlock(cassBlock);
            transformationDiagram.ContainsCassBlock = true;

            DataFlow inputAddress = transformationDiagram.AddInput("Address1", 3);
            cassBlock.AddInput(inputAddress, CassInputTypes.PrimaryAddress);

            // 5. Transfer Address2 field
            transformationDiagram.AddInput("Address2", 4);

            // 6. Add City transformation
            DataFlow inputCity = transformationDiagram.AddInput("City", 5);
            cassBlock.AddInput(inputCity, CassInputTypes.CityName);

            // 7. Transfer State field
            transformationDiagram.AddInput("State", 6);

            // 8. Add Zip transformation
            DataFlow inputZip = transformationDiagram.AddInput("Zip", 7);
            cassBlock.AddInput(inputZip, CassInputTypes.ZipCode);

            // 9. Transfer the rest of columns
            transformationDiagram.AddInput("Country", 8);
            transformationDiagram.AddInput("Phone", 9);
            transformationDiagram.AddInput("Email", 10);
            //transformationDiagram.AddInput("Custom1", 11);
            //transformationDiagram.AddInput("Custom2", 12);
            transformationDiagram.AddInput("IsDuplicate", 11);
            transformationDiagram.AddInput("IsVerified", 12);

            // 10. Prepare diagram
            transformationDiagram.CreateOutputs();
            transformationDiagram.Prepare();

            // 11. Create transformed output storage
            OnDriveTable bufferTransformed = new OnDriveTable(dataPath, transformedTableName);

            for (Int32 i = 0; i < transformationDiagram.Outputs.Count; i++)
            {
                String columnName = transformationDiagram.Outputs[i].Name;
                bufferTransformed.AddField(columnName);
            }

            // 12. Execute standardization with degined before rules
            Int32 rowCount = bufferTable.RecordCount;

            for (Int32 rowIndex = 0; rowIndex < rowCount; rowIndex++)
            {
                transformationDiagram.Process(rowIndex);

                for (Int32 i = 0; i < transformationDiagram.Outputs.Count; i++)
                {
                    String colName = transformationDiagram.Outputs[i].Name;
                    Object value = transformationDiagram.Outputs[i].Value;
                    bufferTransformed.SetData(value, currentRowIndex, colName);
                }

                currentRowIndex++;
            }

            // 13. Show cleansed table
            Int32[] widths1 = ConsoleHelper.GetBestColumnWidths(bufferTransformed, 30);
            ConsoleHelper.PrintTable(bufferTransformed, 20, Int32.MaxValue, widths1);

            ////////////////////////////////////////////////
            //    Matching Part 
            ////////////////////////////////////////////////

            // 14. Create Match Settings - define paths and working subdirectory
            MatchSettings matchSettings = new MatchSettings();
            matchSettings.DataPath = DemoDataPath;
            matchSettings.TempPath = DemoTempPath;
            matchSettings.Name = "Test2Tables";

            // 15. Define matching rules 

            #region Define matching rules 

            // Creating a class responsible for storing of available input fields from data sources, fields mapping, match criterias.
            MultipleMatchDefinitionsManager matchDefinitionManager = new MultipleMatchDefinitionsManager();
            matchDefinitionManager.AvailableFields = new AvailableFields();

            // It's necessary to add available fields from each input table.
            AvailableFieldsFromOneTable fieldsFromTable = new AvailableFieldsFromOneTable();
            fieldsFromTable.Table = inputTable;

            FieldMapInfo fmiFirstName = new FieldMapInfo(0)
            {
                FieldName = "FirstName",
                TableName = sourceTableName,
                ColumnTransformation = null,
                FieldIndex = 1
            };
            fieldsFromTable.Add(fmiFirstName);

            FieldMapInfo fmiLastName = new FieldMapInfo(0)
            {
                FieldName = "LastName",
                TableName = sourceTableName,
                ColumnTransformation = null,
                FieldIndex = 2
            };
            fieldsFromTable.Add(fmiLastName);

            FieldMapInfo fmiAddress1 = new FieldMapInfo(0)
            {
                FieldName = "Address1",
                TableName = sourceTableName,
                ColumnTransformation = null,
                FieldIndex = 3
            };
            fieldsFromTable.Add(fmiAddress1);

            FieldMapInfo fmiZip = new FieldMapInfo(0)
            {
                FieldName = "Zip",
                TableName = sourceTableName,
                ColumnTransformation = null,
                FieldIndex = 7
            };
            fieldsFromTable.Add(fmiZip);

            // Adding fields from single table
            matchDefinitionManager.AvailableFields.TableList.Add(fieldsFromTable);


            AvailableFieldsFromOneTable fieldsFromTable2 = new AvailableFieldsFromOneTable();
            fieldsFromTable2.Table = bufferTransformed;

            FieldMapInfo fmiFirstName1 = new FieldMapInfo(1)
            {
                FieldName = "FirstName",
                TableName = transformedTableName,
                ColumnTransformation = null,
                FieldIndex = 1
            };
            fieldsFromTable2.Add(fmiFirstName1);

            FieldMapInfo fmiLastName1 = new FieldMapInfo(1)
            {
                FieldName = "LastName",
                TableName = transformedTableName,
                ColumnTransformation = null,
                FieldIndex = 4
            };
            fieldsFromTable2.Add(fmiLastName1);

            FieldMapInfo fmiAddress11 = new FieldMapInfo(1)
            {
                FieldName = "Address1",
                TableName = transformedTableName,
                ColumnTransformation = null,
                FieldIndex = 5
            };
            fieldsFromTable2.Add(fmiAddress11);

            FieldMapInfo fmiZip1 = new FieldMapInfo(1)
            {
                FieldName = "Zip",
                TableName = transformedTableName,
                ColumnTransformation = null,
                FieldIndex = 19
            };
            fieldsFromTable2.Add(fmiZip1);

            // Map fields from source 1
            MappedFieldsRow mfrFirstName = new MappedFieldsRow();
            mfrFirstName[sourceTableName] = fmiFirstName;
            mfrFirstName[transformedTableName] = fmiFirstName1;
            matchDefinitionManager.AvailableFields.MappedFieldsRowList.Add(mfrFirstName);

            MappedFieldsRow mfrLastName = new MappedFieldsRow();
            mfrLastName[sourceTableName] = fmiLastName;
            mfrLastName[transformedTableName] = fmiLastName1;
            matchDefinitionManager.AvailableFields.MappedFieldsRowList.Add(mfrLastName);

            MappedFieldsRow mfrAddress1 = new MappedFieldsRow();
            mfrAddress1[sourceTableName] = fmiAddress1;
            mfrAddress1[transformedTableName] = fmiAddress11;
            matchDefinitionManager.AvailableFields.MappedFieldsRowList.Add(mfrAddress1);

            MappedFieldsRow mfrZip = new MappedFieldsRow();
            mfrZip[sourceTableName] = fmiZip;
            mfrZip[transformedTableName] = fmiZip1;
            matchDefinitionManager.AvailableFields.MappedFieldsRowList.Add(mfrZip);

            matchDefinitionManager.SetAbsoluteIndices();

            // Create single Match Definition
            Int32 matchDefinitionIndex = 0;
            MatchCriteriaList matchCriteriaList = new MatchCriteriaList(matchDefinitionIndex);

            // Create fuzzy criteria
            MatchCriteria firstNameCriteria = new MatchCriteria
            {
                Fuzzy = true,
                AddWeightToFirstLetter = false,
                Exact = false,
                Numeric = false,
                UseMetaphone = false,
                IgnoreCase = true,
                Level = 0.9f,
                GroupLevel = 0.0f,
                MinAllowedLevelInGroup = 0.0f,
                GroupId = -1,
                CrossColumnGroupId = -1,
                Weight = 100.0f,
                MatchingIndex = 0,
                AbsoluteMatchingIndex = 0
            };

            Console.WriteLine("Mathc Type: \t\t {1}{0}Fuzzy Level: \t\t {2}", System.Environment.NewLine, "Fuzzy", firstNameCriteria.Level);

            // add field mapping for criteria
            firstNameCriteria.MapField(sourceTableName, "FirstName");
            firstNameCriteria.MapField(transformedTableName, "FirstName");
            matchCriteriaList.Add(firstNameCriteria);



            // Create fuzzy criteria
            MatchCriteria lastNameCriteria = new MatchCriteria
            {
                Fuzzy = true,
                AddWeightToFirstLetter = false,
                Exact = false,
                Numeric = false,
                UseMetaphone = false,
                IgnoreCase = true,
                Level = 0.9f,
                GroupLevel = 0.0f,
                MinAllowedLevelInGroup = 0.0f,
                GroupId = -1,
                CrossColumnGroupId = -1,
                Weight = 100.0f,
                MatchingIndex = 0,
                AbsoluteMatchingIndex = 0
            };

            Console.WriteLine("Mathc Type: \t\t {1}{0}Fuzzy Level: \t\t {2}", System.Environment.NewLine, "Fuzzy", lastNameCriteria.Level);

            // add field mapping for criteria
            lastNameCriteria.MapField(sourceTableName, "LastName");
            lastNameCriteria.MapField(transformedTableName, "LastName");
            matchCriteriaList.Add(lastNameCriteria);



            // Create fuzzy criteria
            MatchCriteria primaryAddressCriteria = new MatchCriteria
            {
                Fuzzy = true,
                AddWeightToFirstLetter = false,
                Exact = false,
                Numeric = false,
                UseMetaphone = false,
                IgnoreCase = true,
                Level = 0.9f,
                GroupLevel = 0.0f,
                MinAllowedLevelInGroup = 0.0f,
                GroupId = -1,
                CrossColumnGroupId = -1,
                Weight = 100.0f,
                MatchingIndex = 0,
                AbsoluteMatchingIndex = 0
            };

            Console.WriteLine("Mathc Type: \t\t {1}{0}Fuzzy Level: \t\t {2}", System.Environment.NewLine, "Fuzzy", primaryAddressCriteria.Level);

            // add field mapping for criteria
            primaryAddressCriteria.MapField(sourceTableName, "Address1");
            primaryAddressCriteria.MapField(transformedTableName, "Address1");
            matchCriteriaList.Add(primaryAddressCriteria);


            // Create fuzzy criteria
            MatchCriteria zipCriteria = new MatchCriteria
            {
                Fuzzy = true,
                AddWeightToFirstLetter = false,
                Exact = false,
                Numeric = false,
                UseMetaphone = false,
                IgnoreCase = true,
                Level = 0.9f,
                GroupLevel = 0.0f,
                MinAllowedLevelInGroup = 0.0f,
                GroupId = -1,
                CrossColumnGroupId = -1,
                Weight = 100.0f,
                MatchingIndex = 0,
                AbsoluteMatchingIndex = 0
            };

            Console.WriteLine("Match Type: \t\t {1}{0}Fuzzy Level: \t\t {2}", System.Environment.NewLine, "Fuzzy", zipCriteria.Level);

            // add field mapping for criteria
            zipCriteria.MapField(sourceTableName, "Zip");
            zipCriteria.MapField(transformedTableName, "Zip");
            matchCriteriaList.Add(zipCriteria);


            matchCriteriaList.MarkTheFirstFieldInEveryGroup();

            matchDefinitionManager.Add(matchCriteriaList);
            //matchDefinitionManager.SetAbsoluteIndices();
            #endregion
            
            // 16. Matching
            
            // Finish init settings
            matchSettings.MatchDefinitionManager = matchDefinitionManager;
            matchSettings.Tables.Add(inputTable);
            matchSettings.Tables.Add(bufferTransformed);
            matchSettings.SourcePairs.Add(new MatchEngine.DataSourceIndexPair(0, 1));
            Console.WriteLine();

            // Create a match engine wrapper. Load inside it settings that were defined in previous steps.
            Console.WriteLine("Creating Matching Manager...");
            MatchManager matchManager = new MatchManager(matchSettings);

            // Run matching
            Console.WriteLine("Matching...");
            Console.WriteLine();
            matchManager.FindMatches();

            // 17. Show result of serching similar records

            OnDriveTable groupsTable = matchSettings.FinalScoresGroupsTable;
            OnDriveTable pairsTable = matchSettings.PairsScoresTable;

            Console.WriteLine("Pairs Table:");
            ConsoleHelper.PrintTable(pairsTable, 10 );

            Console.WriteLine("Groups Table:");
            ConsoleHelper.PrintTable(groupsTable, 8 );

            // 18. Dispose resources

            inputTable.ToDeleteFilesAfterClosing = true;
            inputTable.Dispose();
            inputTable = null;

            bufferTable.ToDeleteFilesAfterClosing = true;
            bufferTable.Dispose();
            bufferTable = null;

            bufferTransformed.ToDeleteFilesAfterClosing = true;
            bufferTransformed.Dispose();
            bufferTransformed = null;

        }

        /// <summary>
        /// Check registration
        /// </summary>
        /// <param name="registrationFilePath">Folder where registration file is kept</param>
        /// <returns>true - if registration is valid </returns>
        private static Boolean InitRegistration(String registrationFilePath)
        {
            Boolean result = false;

            Console.WriteLine("Initializing registration module...");

            _registration.CustomPathForRegistrationFile = CustomPathForRegistrationFile;
            DateTime expirationDate = RegistrationWrapper.ExpirationDate;
            Console.WriteLine(Registration.Registered ? "Registered" : "Not Registered");

            if (expirationDate >= DateTime.Now)
            {
                result = true;
                Console.WriteLine("Initialized Successfully");
            }
            else
            {
                Console.WriteLine("Registration Expired");
            }

            Console.WriteLine("Key expiration date: {0}", expirationDate);
            Console.WriteLine();

            return result;
        }
    }
}
