﻿using System;
using System.IO;
using dataladder.Data;
using dataladder.Matching;
using Samples.Common;
using Samples.Common.Matching;
using dataladder.Licensing;
using dataladder.IO;

namespace Samples.Matching
{
    /// <summary>
    /// Simple example how find duplicates in data sources
    /// </summary>
    class Program
    {
        // path to registration path
        private const String CustomPathForRegistrationFile = @"D:\DataMatchSDK\License";
        // path to wrking folder
        private const String DemoDataPath = @"D:\DataMatchSDK\Test";
        // path to temrorary folder
        private const String DemoTempPath = @"D:\DataMatchSDK\Test\Temp";

        // Several samples can use this object.
        private static RegistrationWrapper _registration = new RegistrationWrapper();

        static void Main(string[] args)
        {
            TestMatchManager();
            
            Console.WriteLine("Press any key to continue . . .");
            Console.ReadKey();
        }

        /// <summary>
        /// Shows initialization, running of Match Engine; shows match results.
        /// </summary>
        public static void TestMatchManager()
        {
            String inputTableName = "Source1";
            String firstNameColName = "FirstName";
            String lastNameColName = "LastName";
            Int32 source1Index = 0;

            if (!InitRegistration(CustomPathForRegistrationFile))
            {
                return;
            }

            // Create Match Settings - define paths and working subdirectory
            MatchSettings matchSettings = new MatchSettings();
            matchSettings.DataPath = DemoDataPath;
            matchSettings.TempPath = DemoTempPath;
            matchSettings.Name = "TestEngine";

            ConsoleHelper.PrintMatchSettings(matchSettings);

            String workingDirectory = Path.Combine(matchSettings.DataPath, matchSettings.Name);

            IOHelper.CreateDirectoryIfNotExist(DemoDataPath);
            IOHelper.CreateDirectoryIfNotExist(DemoTempPath);
            IOHelper.CreateDirectoryIfNotExist(workingDirectory);

            // Creating input table programmatically just for demo purposes. It can be imported from any data source.
            OnDriveTable onDriveTable = CreateDemoTable(workingDirectory, inputTableName);
            ConsoleHelper.PrintTable(onDriveTable, 20);

            // 2. Init Match Definition Manager
            Console.WriteLine("Creating match definitions...");

            // Creating a class responsible for storing of available input fields from data sources, fields mapping, match criterias.
            MultipleMatchDefinitionsManager matchDefinitionManager = new MultipleMatchDefinitionsManager();
            matchDefinitionManager.AvailableFields = new AvailableFields();

            // It's necessary to add available fields from each input table.
            AvailableFieldsFromOneTable fieldsFromTable = new AvailableFieldsFromOneTable();
            fieldsFromTable.Table = onDriveTable;

            FieldMapInfo fmiFirstName = new FieldMapInfo(source1Index)
            {
                FieldName = firstNameColName,
                TableName = inputTableName,
                ColumnTransformation = null,
                FieldIndex = 0
            };
            fieldsFromTable.Add(fmiFirstName);

            FieldMapInfo fmiLastName = new FieldMapInfo(source1Index)
            {
                FieldName = lastNameColName,
                TableName = inputTableName,
                ColumnTransformation = null,
                FieldIndex = 1
            };
            fieldsFromTable.Add(fmiLastName);

            // Adding fields from single table
            matchDefinitionManager.AvailableFields.TableList.Add(fieldsFromTable);

            // Map fields
            MappedFieldsRow mfrFirstName = new MappedFieldsRow();
            mfrFirstName[inputTableName] = fmiFirstName;
            matchDefinitionManager.AvailableFields.MappedFieldsRowList.Add(mfrFirstName);

            MappedFieldsRow mfrLastName = new MappedFieldsRow();
            mfrLastName[inputTableName] = fmiLastName;
            matchDefinitionManager.AvailableFields.MappedFieldsRowList.Add(mfrLastName);

            matchDefinitionManager.SetAbsoluteIndices();

            // Create single Match Definition
            Int32 matchDefinitionIndex = 0;
            MatchCriteriaList matchCriteriaList = new MatchCriteriaList(matchDefinitionIndex);

            // Create fuzzy criteria
            MatchCriteria firstNameCriteria = new MatchCriteria
            {
                Fuzzy = true,
                AddWeightToFirstLetter = false,
                Exact = false,
                Numeric = false,
                UseMetaphone = false,
                IgnoreCase = true,
                Level = 0.7f,
                GroupLevel = 0.0f,
                MinAllowedLevelInGroup = 0.0f,
                GroupId = -1,
                CrossColumnGroupId = -1,
                Weight = 100.0f,
                MatchingIndex = 0,
                AbsoluteMatchingIndex = 0
            };

            Console.WriteLine("Mathc Type: \t\t {1}{0}Fuzzy Level: \t\t {2}", System.Environment.NewLine, "Fuzzy", firstNameCriteria.Level);

            // add field mapping for criteria
            firstNameCriteria.MapField(inputTableName, firstNameColName);

            matchCriteriaList.Add(firstNameCriteria);
            matchCriteriaList.MarkTheFirstFieldInEveryGroup();

            matchDefinitionManager.Add(matchCriteriaList);
            //matchDefinitionManager.SetAbsoluteIndices();

            // 3. Matching

            // Finish init settings
            matchSettings.MatchDefinitionManager = matchDefinitionManager;
            matchSettings.Tables.Add(onDriveTable);
            matchSettings.SourcePairs.Add(new MatchEngine.DataSourceIndexPair(source1Index, source1Index));

            // matching
            Console.WriteLine();
            Console.WriteLine("Creating Matching Manager...");
            MatchManager matchManager = new MatchManager(matchSettings);

            Console.WriteLine("Matching...");
            Console.WriteLine();
            matchManager.FindMatches();

            //4. Show results and free resources

            // results
            OnDriveTable groupsTable = matchSettings.FinalScoresGroupsTable;
            OnDriveTable pairsTable = matchSettings.PairsScoresTable;

            // show results
            Console.WriteLine("Pairs Table:");
            ConsoleHelper.PrintTable(pairsTable, 10);

            Console.WriteLine("Groups Table:");
            ConsoleHelper.PrintTable(groupsTable, 10);

            onDriveTable.ToDeleteFilesAfterClosing = true;
            onDriveTable.Dispose();
            onDriveTable = null;
        }

        /// <summary>
        /// Check registration
        /// </summary>
        /// <param name="registrationFilePath">folder where registration file is kept</param>
        /// <returns>true if registration is valid</returns>
        private static Boolean InitRegistration(String registrationFilePath)
        {
            Boolean result = false;

            Console.WriteLine("Initializing registration module...");

            _registration.CustomPathForRegistrationFile = CustomPathForRegistrationFile;
            DateTime expirationDate = RegistrationWrapper.ExpirationDate;

            if (expirationDate >= DateTime.Now)
            {
                result = true;
                Console.WriteLine("Initialized Successfully");
            }
            else
            {
                Console.WriteLine("Registration Expired");
            }

            Console.WriteLine("Key expiration date: {0}", expirationDate);
            Console.WriteLine();

            return result;
        }

        /// <summary>
        /// Create simple filled storage
        /// </summary>
        /// <param name="path">folder where new storege will be cretaed</param>
        /// <param name="name">name of storage</param>
        /// <returns></returns>
        private static OnDriveTable CreateDemoTable(String path, String name)
        {
            // 1. Init Data Source
            Console.WriteLine("Creating source table: {0}", name);

            OnDriveTable onDriveTable = new OnDriveTable(path, name);
            onDriveTable.AddField("FirstName", typeof(String));
            onDriveTable.AddField("LastName", typeof(String));
            onDriveTable.SetData("John", 0, 0);
            onDriveTable.SetData("Smith", 0, 1);
            onDriveTable.SetData("Joan", 1, 0);
            onDriveTable.SetData("Smit", 1, 1);

            return onDriveTable;
        }
    }
}
