﻿using System;
using System.Collections.Generic;
using dataladder.Data;
using dataladder.Data.DataTransformation;
using dataladder.AddressVerification;
using dataladder.Licensing;
using DataMatch.AddressVerification.Cass;
using Samples.Common;

namespace Samples.Transformation
{
    /// <summary>
    /// Simple example with address verification
    /// </summary>
    internal class Program
    {
        // Path to DME API license key (this key is different from a simple DME Desktop application key)
        private const String CustomPathForRegistrationFile = @"D:\DataMatchSDK\License";

        // Working & temporary directories for the current sample.
        private const String DemoDataPath = @"D:\DataMatchSDK\Test\TestClean";
        private const String TempDataPath = @"D:\DataMatchSDK\Test\Temp";
        private const String UncompleteFolder = "uncomplete";

        // Input and transformed table names
        private const String InputTableName = @"Source1";
        private const String CleanedTableName = @"Transformed";

        // Paths for address verification DBs
        private const String AddressDbPath = @"C:\enterprise";
        private const String GeoDbPath = @"C:\enterprise\Geocoder\Accugeo";

        // For single-threaded application the next value is not important.
        private const Int32 TaskIndex = 0;

        // Number of cores to use during transformation. This value is ignored atm.
        private const Int32 PartsCountToUse = 0;

        // Several samples can use this object.
        private static readonly RegistrationWrapper Registration = new RegistrationWrapper();

        // Required for address transformation.
        private static readonly CassParser CassModule = new CassParser();

        static void Main(string[] args)
        {
            TestTransformation();

            Console.WriteLine(@"Press any key to continue . . .");
            Console.ReadKey();
        }

        /// <summary>
        /// Core of example.
        /// - Check registration
        /// - Create simple data
        /// - Create standardization settings
        /// - Clean dat
        /// - Show results
        /// </summary>
        public static void TestTransformation()
        {
            if (!InitRegistration())
            {
                return;
            }

            InitCassModule();

            dataladder.IO.IOHelper.CreateDirectoryIfNotExist(DemoDataPath);
            dataladder.IO.IOHelper.CreateDirectoryIfNotExist(TempDataPath);

            OnDriveTable inputTable = CreateAddressTable(DemoDataPath, InputTableName);

            ConsoleHelper.PrintTable(inputTable, 20);

            TransformationDiagram transformationDiagram = CreateTransformationDiagram(inputTable);

            OnDriveTable outputTable = TransformTable(inputTable, transformationDiagram);

            ConsoleHelper.PrintTable(outputTable, 20);

        }

        /// <summary>
        /// Init registration module
        /// </summary>
        /// <returns></returns>
        private static Boolean InitRegistration()
        {
            Console.WriteLine(@"Initializing registration module...");

            Registration.CustomPathForRegistrationFile = CustomPathForRegistrationFile;
            DateTime expirationDate = RegistrationWrapper.ExpirationDate;
            Boolean registered = expirationDate >= DateTime.Now;

            Console.WriteLine(registered ? @"Initialized Successfully" : @"Registration Expired");
            Console.WriteLine(@"Key expiration date: {0}", expirationDate);
            Console.WriteLine();

            return registered;
        }

        /// <summary>
        /// Init address verification module
        /// </summary>
        private static void InitCassModule()
        {
            Console.WriteLine(@"Initializing CASS...");
            CassModule.InitCassIfNeeded(AddressDbPath, GeoDbPath);
        }

        /// <summary>
        /// Creates hardcoded standardization settings that are based on specified data storage
        /// </summary>
        /// <param name="inputTable">storage</param>
        /// <returns>standardization scheme</returns>
        private static TransformationDiagram CreateTransformationDiagram(OnDriveTable inputTable)
        {
            AddressVerificationSettings addressVerificationSettings = CreateAddressVerificationSettings();
            AddressCassBlock cassBlock = new AddressCassBlock(CassModule.CassAddress, CassModule.CassGeoCoder, addressVerificationSettings);

            List<Int32> fieldIndexMapping = new List<Int32> { 1, 2, 3, 4 };

            TransformationDiagram transformationDiagram = new TransformationDiagram(TaskIndex);
            transformationDiagram.OnGetInputData += (index, taskIndex, rowIdx) => inputTable.GetData(rowIdx, fieldIndexMapping[index]);
            transformationDiagram.Clear();
            transformationDiagram.AddTransformationBlock(cassBlock);
            transformationDiagram.ContainsCassBlock = true;

            DataFlow inputAddress = transformationDiagram.AddInput("Address", 0);
            cassBlock.AddInput(inputAddress, CassInputTypes.PrimaryAddress);
            DataFlow inputCity = transformationDiagram.AddInput("City", 1);
            cassBlock.AddInput(inputCity, CassInputTypes.CityName);
            DataFlow inputState = transformationDiagram.AddInput("State", 2);
            cassBlock.AddInput(inputState, CassInputTypes.StateName);
            DataFlow inputZip = transformationDiagram.AddInput("Zip", 3);
            cassBlock.AddInput(inputZip, CassInputTypes.ZipCode);

            transformationDiagram.CreateOutputs();
            transformationDiagram.Prepare();

            return transformationDiagram;
        }

        /// <summary>
        /// Make cleansing of specified storage
        /// </summary>
        /// <param name="inputTable">dirty storage</param>
        /// <param name="transformationDiagram">standardization settings</param>
        /// <returns>storage with cleansed results</returns>
        private static OnDriveTable TransformTable(OnDriveTable inputTable, TransformationDiagram transformationDiagram)
        {
            OnDriveTable outputTable = new OnDriveTable(DemoDataPath, CleanedTableName);

            for (Int32 i = 0; i < transformationDiagram.Outputs.Count; i++)
            {
                String columnName = transformationDiagram.Outputs[i].Name;
                outputTable.AddField(columnName);
            }

            Int32 rowCount = inputTable.RecordCount;

            for (Int32 rowIndex = 0; rowIndex < rowCount; rowIndex++)
            {
                transformationDiagram.Process(rowIndex);

                for (Int32 i = 0; i < transformationDiagram.Outputs.Count; i++)
                {
                    String colName = transformationDiagram.Outputs[i].Name;
                    Object value = transformationDiagram.Outputs[i].Value;
                    outputTable.SetData(value, rowIndex, colName);
                }
            }

            return outputTable;
        }

        /// <summary>
        /// Creates simple storage and adds to it one record
        /// </summary>
        /// <param name="path"></param>
        /// <param name="name"></param>
        /// <returns></returns>
        private static OnDriveTable CreateAddressTable(String path, String name)
        {
            // 1. Init Data Source
            Console.WriteLine(@"Creating source table: {0}", name);
            
            // create
            OnDriveTable onDriveTable = new OnDriveTable(path, name);
            onDriveTable.AddField("FullName", typeof(String));
            onDriveTable.AddField("Address", typeof(String));
            onDriveTable.AddField("City", typeof(String));
            onDriveTable.AddField("State", typeof(String));
            onDriveTable.AddField("Zip", typeof(String));

            // fill first row
            onDriveTable.SetData("John Smith", 0, 0);
            onDriveTable.SetData("68 Bridge St. Suite 304", 0, 1);
            onDriveTable.SetData("Suffield", 0, 2);
            onDriveTable.SetData("CT", 0, 3);
            onDriveTable.SetData("06078", 0, 4);

            return onDriveTable;
        }

        /// <summary>
        /// Defines which columns will be created after address verification
        /// </summary>
        /// <returns></returns>
        private static AddressVerificationSettings CreateAddressVerificationSettings()
        {
            AddressVerificationSettings addressVerificationSettings = new AddressVerificationSettings();
            addressVerificationSettings.OutputColumnSettings.ForEach(s =>
            {
                switch (s.CassOutputPart)
                {
                    case CassOutputParts.Status:
                    case CassOutputParts.PrimaryAddress:
                    case CassOutputParts.City:
                    case CassOutputParts.State:
                    case CassOutputParts.ZipCode:
                    case CassOutputParts.Latitude:
                    case CassOutputParts.Longitude:
                        s.Include = true;
                        break;
                    default:
                        s.Include = false;
                        break;
                }
            });

            return addressVerificationSettings;
        }

    }
}
